/*******************************************
list.c
GCLUST
previously a part of lsort3.h 
This program reads a list file (generated from BLAST result by bl2ls.pl),
and assembles similarity groups. Output is a *.grp file.
This is a C version of the lsort2b.pl.
  Copyright Naoki Sato 2002.

Added functionality of using catenated list file. June 2, 2002.
Added msort, lnkann. June 16, 2002.
Added SQlist. June 17, 2002.
Added lnkdb. June 25, 2002.
Selection of master entry. June 25, 2002.
Added hom. July 2, 2002.
Matrix is now in float. July 5, 2002.
Matrix is now in int. July 6, 2002.
score is now in float. July 6, 2002. Now double, May 13, 2003.
Added sublist5. July 7, 2002.
Phase 1. July 19, 2002.
RemoveUnrelated returns unsigned. Sept. 5, 2002.
Unnecessary output suppressed. Dec. 30, 2002.
Reconstruction of source code. April 30, 2003.
Bug fix in cmpSQlist. May 15, 2003.
Added InitClist and ClearClist. Sept. 24, 2003.
Bug fix in recoverIDlist. Sept. 17, 2004.
Added ListRelatives. June 7, 2005.
Added RepeatedDomains. July 4, 2005.
Added MergeDistantRelatives. Sept. 6, 2005.
The following two modifications change clustering behavior of small proteins.
Modified MergeDistantRelatives. June 26, 2007.
Modified BinaryInteractions. June 27, 2007.
*******************************************/

#include "defines.h"

void MergeDistantRelatives(Node **a,unsigned nodes);
void LimitScore(Node **a, unsigned nodes);
void reassignDMlist(Node **a, unsigned nodes, unsigned i);
char *DomainInfo(Node **a, unsigned nodes, unsigned i, unsigned j);
char *RepeatedDomains(Node **a, unsigned nodes, unsigned i);
void ListRelatives(Node **a, unsigned nodes);
void AddGrpno(Node **a, unsigned nodes);
void SortRelatives(Node **a, unsigned nodes);
Node **AddList2(Node **a,unsigned nodes,FILE *missing);
Node **CleanList(Node **a,unsigned nodes);
Node **MergeList(Node **a,unsigned nodes);
Node **CreateCliques(Node **a,unsigned nodes);
void MakeClist(Node **a, unsigned nodes);
Node **MergeCliques(Node **a,unsigned nodes);
Node **BinaryInteractions(Node **a,unsigned nodes);
void CleanupIDall(Node **a,unsigned nodes);
void CleanupID(Node **a,unsigned nodes,unsigned i);
void RemoveID(Node **a,unsigned nodes,unsigned i,unsigned ID);
void MakeIDlist_i(Node **a,unsigned nodes,unsigned i);
void MakeIDlist_clique(Node **a,unsigned nodes,unsigned i);
void MakeIDlist(Node **a,unsigned nodes);
void clearSQlist(SQlist *A);
void copySQlist(SQlist *A, SQlist *B);
void RcopySQlist(SQlist *A, SQlist *B);
int cmpSQlist(SQlist *A, SQlist *B, double level1, double level2);
void MakeSublist(Node **a,unsigned nodes,unsigned i);
void WriteParent(Node **a,unsigned nodes);
void RenewList6(Node **a,unsigned nodes,unsigned k);
int cmpIDlist(Node **a,unsigned nodes,int it);
void SearchMultiDomain(Node **a,unsigned nodes,unsigned i);
int RemoveMultiDomain(Node **a,unsigned nodes,unsigned i);
void InitStack(Stack *stack1,unsigned size);
void CopyStack(Stack *stack1,Stack *stack2);
int RegionTest_idlist4(Node **a,unsigned nodes,unsigned i);
void recoverIDlist(Node **a,unsigned nodes,unsigned i);
void backupIDlist(Node **a,unsigned nodes,unsigned i);
void clearIDlist(Node **a,unsigned nodes,unsigned i);
void clearN3list(Node **a,unsigned nodes);
DMlist*expandDMlist(DMlist *dmlist_in,unsigned n,unsigned m);
void SetMultiDomain(Node **a,unsigned nodes,unsigned n);
void InitClist(unsigned nodes);
void ClearClist(unsigned nodes);
unsigned ShowHighestBit(unsigned long ID);
unsigned CountBits(unsigned long ID);
void ReassignDomainID(Node **a,unsigned nodes,unsigned i);

#ifndef __LIST__
#define __LIST__
#endif
#include "gclust.h"



/*********************************************************/
void LimitScore(Node **a,unsigned nodes)
/*********************************************************/
{
	unsigned i,j,n1;
	double thr_1,thr_2;
	unsigned ID2,len,len2;
	Boolean to_delete=FALSE;

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if((n1=a[i]->n1)==0) continue;
		len = a[i]->len;
		thr_1 = limit_thr(len);
		if(len >= 100) len = 100;
		for(j=0;j<n1;j++){
			if(a[i]->sqlist1[j].score > thr_1){
				ID2 = a[i]->sqlist1[j].ID;
				if(ID2 == 0 || ID2 > nodes) continue;
				if(ID2 == i) continue;
/*				ID2 = FindNode(a,nodes,a[i]->sqlist1[j].ID);
*/
				to_delete = TRUE;
				if((len2=a[ID2]->len) < len){
					thr_2 = limit_thr(len2);
					if(a[i]->sqlist1[j].score <= thr_2) to_delete = FALSE;
				}
				if(to_delete){
					swapSQlist(&(a[i]->sqlist1[j]),&(a[i]->sqlist1[n1-1]));
					n1--;
					j--;
				}
			}
		}
		a[i]->n1 = n1;
	}
	return;
}
/*********************************************************/


/*********************************************************/
void reassignDMlist(Node **a, unsigned nodes, unsigned i)
/*********************************************************/
{
	unsigned m;
	unsigned long dmID = 0x01;

	for(m=a[i]->DMnum;m>0;m--){
		a[i]->dmlist[m-1].ID = dmID;
		dmID <<= 1;
	}
	return;
}
/*********************************************************/


/*********************************************************/
char *DomainInfo(Node **a, unsigned nodes, unsigned i, unsigned j)
/*********************************************************/
{
	static char *dmstring;
	unsigned k, m, n, t1;
	unsigned long dmID,dmID2;
	unsigned bits;
	unsigned a1,a2,b1,b2;
	double x;
	double level1 = 0.8;
	double level2 = 0.5;

	if(dmstring==NULL){
		if((dmstring=(char*)calloc(100,sizeof(char)))==NULL){
			fprintf(stderr,"Momory allocation error in DomainInfo.\n");
			exit(1);
		}
	} else {
		strcpy(dmstring,"");
	}

	t1 = a[i]->n3;
	if(t1 == 0) t1 = a[i]->n3b;
	if(t1 == 0) fprintf(stderr,"n3 = 0 for i = %u.\n",i);
	
	n = a[i]->relatives[j].ID;
/*
printf("DomainInfo for node=%u, group=%u, t1=%u, n=%u.\n",a[i]->n0,a[i]->grpno,t1,n);
*/
	dmID = 0x00;
	dmID2 = 0x00;
	for(k=0;k<t1;k++){
		if(n == a[i]->sqlist3[k].ID && a[i]->sqlist3[k].qID == a[i]->n0){
			b1=a[i]->sqlist3[k].Qstart;
			b2=a[i]->sqlist3[k].Qend;
			for(m=0;m<a[i]->DMnum;m++){
				a1=a[i]->dmlist[m].start;
				a2=a[i]->dmlist[m].end;
				x = overlap(a1,a2,b1,b2);
/*
printf("a1=%u, a2=%u, b1=%u, b2=%u, x=%f\n",a1,a2,b1,b2,x);
*/				
				if(((double)x / (double)(a2 - a1)) >= level1){
					dmID |= a[i]->dmlist[m].ID;
				} else if(((double)x / (double)(a2 - a1)) >= level2){
					dmID2 |= a[i]->dmlist[m].ID;
				}
			}
		}
	}
	bits = a[i]->DMnum;	

	strcpy(dmstring, c10_3(dmID,dmID2,bits,'8','5'));

	return dmstring;
}
/*********************************************************/


/*********************************************************/
char *RepeatedDomains(Node **a, unsigned nodes, unsigned i)
/* To estimate repeated domains, similarity data with 
   other homologs are used. */
/*********************************************************/
{
	static char *dmstring;
	unsigned k, m, n, t1,s;
	unsigned m1,m2;
	unsigned j,nR;
	int dmIDstring[50];
	int dmIDstring2[50];
	int dmIDstring3[50];
	int counter;
	unsigned a1,a2,b1,b2;
	double x;
	double level1 = 0.8;
	int dmIDs[50][50];	/* maximum size is limited to 50 */
	Boolean to_exit = FALSE;

	if(dmstring==NULL){
		if((dmstring=(char*)calloc(50,sizeof(char)))==NULL){
			fprintf(stderr,"Momory allocation error in RepeatedDomains 1.\n");
			exit(1);
		}
	} else {
		strcpy(dmstring,"");
	}
	for(m=0;m<50;m++){
		dmIDstring3[m] = 0;
	}

	t1 = a[i]->n3;
	if(t1 == 0) t1 = a[i]->n3b;
	if(t1 == 0) fprintf(stderr,"n3 = 0 for i = %u.\n",i);

	nR = a[i]->nR;

	for(j=0;j<nR;j++){
		n = a[i]->relatives[j].ID;
		s = FindNode(a,nodes,n);
		if(s == 0 || s > nodes) continue;
		m1 = m2 = 0;
		for(m=0;m<50;m++){
			for(m1=0;m1<50;m1++){
				dmIDs[m][m1] = 0;
			}
		}
		
		for(k=0;k<t1;k++){
			if(n == a[i]->sqlist3[k].ID && a[i]->sqlist3[k].qID == a[i]->n0){
				counter = 1;
				to_exit = FALSE;
				for(m=0;m<50;m++){
					dmIDstring[m] = 0;
					dmIDstring2[m] = 0;
				}
				/* identifying domain in entry i */
				b1=a[i]->sqlist3[k].Qstart;
				b2=a[i]->sqlist3[k].Qend;
				for(m=0;m<a[i]->DMnum;m++){
					a1=a[i]->dmlist[m].start;
					a2=a[i]->dmlist[m].end;
					x = overlap(a1,a2,b1,b2);
					if(((double)x / (double)(a2 - a1)) >= level1){
						dmIDstring[m] = counter++;
					}
					if(counter > 3 || counter * 2 > a[i]->DMnum){
						to_exit = TRUE;
						break;
					}
				}
				if(to_exit) continue;
				/* identifying domain in entry n */
				b1=a[i]->sqlist3[k].Sstart;
				b2=a[i]->sqlist3[k].Send;
				for(m=0;m<a[s]->DMnum;m++){
					a1=a[s]->dmlist[m].start;
					a2=a[s]->dmlist[m].end;
					x = overlap(a1,a2,b1,b2);
					if(((double)x / (double)(a2 - a1)) >= level1){
						dmIDstring2[m] = 1;
					}
				}
				for(m=0;m<a[s]->DMnum;m++){
					if(dmIDstring2[m]){
						for(m1=0; m1<a[i]->DMnum;m1++){
							if(dmIDs[m][m1] == 0 || (dmIDstring[m1] > 0 && dmIDs[m][m1] > dmIDstring[m1])){
								 dmIDs[m][m1] = dmIDstring[m1];
							}
						}
					}
				}
			}
		}

		/* searching for repeated domains */
		for(k=1;k<5;k++){
			for(m=0;m<a[s]->DMnum;m++){
				counter = 0;
				for(m1=0;m1<a[i]->DMnum;m1++){
					if(dmIDs[m][m1] == k) counter++;
				}
				if(counter > 1) break;
			}
			if(counter > 1) break;
		}
		if(counter > 1){
			for(m=1;m<a[s]->DMnum;m++){
				for(m1=0;m1<a[i]->DMnum;m1++){
					if(dmIDs[m][m1] == 1) dmIDs[0][m1] = 1;
				}
			}
			for(m=0;m<a[i]->DMnum;m++){
				if(dmIDstring3[m] == 0 || (dmIDs[0][m] > 0 && dmIDstring3[m] > dmIDs[0][m])){
					dmIDstring3[m] = dmIDs[0][m];
				}
			}
		}

	}
	
	for(m=0;m<a[i]->DMnum;m++){
		if(dmIDstring3[m] != 0) break;
	}
	if(m==a[i]->DMnum) {
		strcpy(dmstring,"");
	} else {
		for(m=0,m1=0;m<a[i]->DMnum;m++,m1++){
			if(dmIDstring3[m] > 9) {
				 dmstring[m1] = 'A';
			} else if(dmIDstring3[m] == 0){
				 dmstring[m1] = '.';
			} else {
				 dmstring[m1] = 48 + dmIDstring3[m];
			}
			if((a[i]->DMnum - m - 1) % 4 == 0) dmstring[++m1] = ' ';
		}
		dmstring[m1-1] = '\0';

/*
		printf("i=%u, %s\n",i,dmstring);
*/
	}
	return dmstring;
}
/*********************************************************/



/*********************************************************/
void ListRelatives(Node **a, unsigned nodes)
/* Only in clique mode */
/* Previously, only relatives of the parent node have been
 * listed as relatives. If, however, all relatives are
 * listed for all members, the list will be enormous.
 * From version 355m, exhaustive search for all relatives was 
 * added, excluding very large clusters. Number of edges 
 * is also counted.*/
/*********************************************************/
{
	unsigned i,j,k,m,n,i1,i2;
	unsigned grpno;
	unsigned nR;
	unsigned n7,n3,n4;
	Relatives *r;
	unsigned jj,kk,mm;

	for(i=1;i<=nodes;i++){
		if((n3 = a[i]->n3) == 0) continue;
		if((grpno=a[i]->grpno) == 0) continue;
		n4 = a[i]->n4;
		n7 = a[i]->n7;
		nR = 10;
		if((a[i]->relatives = (Relatives *)calloc(nR,sizeof(Relatives)))==NULL){
			fprintf(stderr,"Memory allocation error in ListRelatives 1.\n");
			exit(1);
		}
		r = a[i]->relatives;
		i1 = 0;

		for(jj=0;jj<n4;jj++){

			if(jj != 0 && i1 > 3 * number_of_genomes && i1 > 5 * n4) break;
			kk = a[i]->idlist4[jj];
			mm = FindNode(a,nodes,kk);

			/* matchlist7 is now produced for domain == 1, too, in AssignSingletons. */
			if(a[mm]->domain == 0 || a[mm]->domain == 1){
				for(j=0;j<n7;j++){
					k = a[mm]->matchlist7[j].ID;
					m = FindNode(a,nodes,k);
					if((n = a[m]->pID) != a[i]->n0 && n != 0){
						for(i2=0;i2<i1;i2++) {
							if(r[i2].pID == n && r[i2].ID == k){
								r[i2].count += 1;		
								break;
							}
						}
						if(i2 < i1) {
							continue;
						}
						if(i1 >= nR) {
							nR += 10;
							if((r = (Relatives *)realloc(r,nR * sizeof(Relatives)))==NULL){
								fprintf(stderr,"Memory allocation error in ListRelatives 3.\n");
								exit(1);
							}
						}
						r[i1].grpno = 0;
						r[i1].pID = n;
						r[i1].count = 1;
						r[i1++].ID = k;

					}
				}
			} else {
				for(j=0;j<n3;j++){
					k = a[mm]->sqlist3[j].ID;
					m = FindNode(a,nodes,k);
					if((n = a[m]->pID) != a[i]->n0 && n != 0){
						for(i2=0;i2<i1;i2++) {
							if(r[i2].pID == n && r[i2].ID == k){
								r[i2].count += 1;		
								break;
							}
						}
						if(i2 < i1) continue;
						if(i1 >= nR) {
							nR += 10;
							if((r = (Relatives *)realloc(r,nR * sizeof(Relatives)))==NULL){
								fprintf(stderr,"Memory allocation error in ListRelatives 4.\n");
								exit(1);
	
							}
						}
						r[i1].grpno = 0;
						r[i1].pID = n;
						r[i1].count = 1;
						r[i1++].ID = k;
					}
				}
			}
		}	/* jj */

		a[i]->relatives = r;
		a[i]->nR = i1;
	}	/* i */

	return;
}
/*********************************************************/


/*********************************************************/
void AddGrpno(Node **a,unsigned nodes)
/*********************************************************/
{
	unsigned i,j,k;
	unsigned nR;
	unsigned pID;

	for(i=1;i<=nodes;i++){
		if(a[i]->n3 == 0) continue;
		nR = a[i]->nR;
		for(j=0;j<nR;j++){
			pID = a[i]->relatives[j].pID;
			k = FindNode(a,nodes,pID);
			/* 352f9 */
			if(k==0) a[i]->relatives[j].grpno = 0;
			else a[i]->relatives[j].grpno = a[k]->grpno;
		}
	}

	return;
}
/*********************************************************/

/*********************************************************/
void SortRelatives(Node **a, unsigned nodes)
/* 355o: Removal of null items */
/*********************************************************/
{
	unsigned i,j,k;
	unsigned nR;
	unsigned grpno,pID,ID,count;

	for(i=1;i<nodes;i++){
		if(a[i]->n3 == 0) continue;
		if((nR = a[i]->nR) == 0) continue;
		for(j=0;j<nR;j++){
			if(a[i]->relatives[j].grpno == 0){
				if(nR == 1){
					nR = 0;
					break;
				} else if(j + 1 == nR){
					nR--;
					break;
				}
				while(a[i]->relatives[nR-1].grpno == 0){
					nR--;
					if(nR <= j + 1) break;
				}
				if(nR == 0) break;
				k = nR - 1;
				if(j < k){
					a[i]->relatives[j].grpno = a[i]->relatives[k].grpno;
					a[i]->relatives[j].pID = a[i]->relatives[k].pID;
					a[i]->relatives[j].ID = a[i]->relatives[k].ID;
					a[i]->relatives[j].count = a[i]->relatives[k].count;
				}
				nR--;
				if(nR == 0) break;
			}
			if(nR == 0){
				a[i]->nR = 0;
				break;
			}
		}
		a[i]->nR = nR;
	}

	for(i=1;i<nodes;i++){
		if(a[i]->n3 == 0) continue;
		if((nR = a[i]->nR) == 0) continue;
		for(j=0;j+1<nR;j++){
			for(k=j+1;k<nR;k++){
				/* 352f9 <= was replaced by < */
				if(a[i]->relatives[j].grpno < a[i]->relatives[k].grpno) continue;
				else if((a[i]->relatives[j].grpno == a[i]->relatives[k].grpno) && \
						(FindNode(a,nodes,a[i]->relatives[j].ID) <= FindNode(a,nodes,a[i]->relatives[k].ID))){
					continue;
				}
				grpno = a[i]->relatives[j].grpno;
				pID = a[i]->relatives[j].pID;
				ID = a[i]->relatives[j].ID;
				count = a[i]->relatives[j].count;
				a[i]->relatives[j].grpno = a[i]->relatives[k].grpno;
				a[i]->relatives[j].pID = a[i]->relatives[k].pID;
				a[i]->relatives[j].ID = a[i]->relatives[k].ID;
				a[i]->relatives[j].count = a[i]->relatives[k].count;
				a[i]->relatives[k].grpno = grpno;
				a[i]->relatives[k].pID = pID;
				a[i]->relatives[k].ID = ID;
				a[i]->relatives[k].count = count;

			}
		}
	}

	return;
}
/*********************************************************/


/*********************************************************/
void MergeDistantRelatives(Node **a,unsigned nodes)
/* In the clique mode, two cliques are merged if the smaller
 * one is largely included in the larger one.
 * This is done at the last stage. Changes are made on
 * only idlist4 but not on sqlist3.
 * (old algorithm)
 * The following criteria are applied in this order, namely
 * descending order of priority.
 * 1. Multidomain or large protein is not processed.
 * 2. Final size of merged cluster is limeted to 2 times 
 *    number of genomes.
 * 3. If the increment in number of organisms is small or
 *    if the overlap in organisms in the two nodes is large,
 *    merging is not attempted.
 * 5. If all relatives belong to a single cluster, and the 
 *    relatives of that cluster belong to the original 
 *    cluster, the two clusters are merged.
 * 4. Relatives are incorporated if more than 0.75 x ns4 IDs
 *    are listed as relatives. */
/* Regroup level 1 (weak) - 10 (hard), 5 is standard.*/ 
/* For small proteins, regroup level is increased. June 2007.
 * Jensen-Shannon divergence measure is used in CountOrgInIDlist.
 * July 2007. */
/* In version 355r, clustering is complete without regroup
 * option. This option is now used to search paralogous
 * groups. For this purpose, the criteria are now more
 * rigorous. Aug. 2007.
 * Cliques of groups are explicitly searched. */
/*********************************************************/
{
	unsigned i,j2,k,m,n,mx,nn;
	unsigned n4, nR1, nR2, ID,IDx, n4x,n4_total;
	int count,count2,j;
	double level1 = 0.75;
	double level2 = _level2;	/* 1.7 */  
	unsigned max_cluster = 2 * number_of_genomes;
	int to_merge = 0;
	/* Version 355, June 2007 */
	double level1_bak = level1;
	unsigned max_cluster_bak = max_cluster;
	unsigned length,length2,length3;
	double score;
	int regroup_level = _regroup_level;
	unsigned *rel_list;	/* first item is pID, second item is occurrence, third item is cross edge */
	unsigned max_rel = 0;
	unsigned max_nR1;
	double cross_edge,cross_edge0,domain_measure;	/* 355n */
	int clique_size = _MDR_clique_size;	/* 355v */
	Boolean to_exit = FALSE;
	int ct;
	char word[20];
	Boolean small_addition = FALSE;
	unsigned SA = num_org/10;
	double cross_edge_parameter = _cross_edge_parameter;

/* 353h standard values:
	double _MDR_level1_0 = 0.95;
	double _MDR_level1_1 = 0.03;
	double _MDR_max_cluster_0 = 1.25;
	double _MDR_max_cluster_1 = 0.10;
*/
/* 355 standard values:
	unsigned _MDR_length1 = 70;
	unsigned _MDR_length2 = 120;
	double _MDR_score1 = 1e-30;
	double _MDR_score2 = 1e-45;

	double cross_edge_parameter = 0.4;
*/
	if(SA < 2) SA =2;
	ct = 0;
	printf("Merging distant relatives: ");
	printf("level1=%f, max_cluster=%u\n",level1,max_cluster);

	/* 353f */
	if(_regroup_level >= 0){
		level1 = _MDR_level1_0 - _MDR_level1_1 * (double)_regroup_level;
		max_cluster = (_MDR_max_cluster_0 + _MDR_max_cluster_1 * (double)_regroup_level) * (double)number_of_genomes;
		level1_bak = level1;
		max_cluster_bak = max_cluster;
	}
	regroup_level = _regroup_level;
	if(_regroup_level > 12) _regroup_level = 10;

	max_nR1 = 0;
	for(i=1;i<=nodes;i++){
		if(a[i]->n3 == 0 || a[i]->n4 == 0) continue;
		if(a[i]->domain != 0) continue;	
		if(max_nR1 < a[i]->nR) max_nR1 = a[i]->nR;
	}

	/* rel_list is defined only once. It is re-used by clearing values. */
	if((rel_list = (unsigned*)(calloc(3 * max_nR1 + 3,sizeof(unsigned))))==NULL){
		fprintf(stderr,"Memory allocation error in MergeDistantRelatives 0.\n");
		exit(1);
	}
	
	for(k=0;k<max_nR1;k++){
		rel_list[3 * k] = 0;
		rel_list[3 * k + 1] = 0;
		rel_list[3 * k + 2] = 0;
	}
	max_rel = 0;

	/* First, a[i]->rel_list is prepared. */
	printf("\nMDR stage 1: rel_list is being prepared ...\n");
	for(i=1;i<=nodes;i++){
		/* The rel_list of all nodes are cleared. */
		a[i]->max_rel = 0;
		if(a[i]->n3 == 0 || a[i]->n4 == 0) continue;
		if(a[i]->domain != 0) continue;		/* criterion 1 */
		nR1 = a[i]->nR;

		_regroup_level = regroup_level;

		/* score of self similarity */
		score = thr_list[0];
		for(j=0;j<a[i]->n3;j++){
			if(a[i]->sqlist3[j].ID == a[i]->n0 && a[i]->sqlist3[j].qID == a[i]->n0){
				if(score > a[i]->sqlist3[j].score) score = a[i]->sqlist3[j].score;
			}
		}
		/* average corrected length */
		length = 0;
		for(j=0;j<a[i]->n4;j++){
			ID = a[i]->idlist4[j];
			n = FindNode(a,nodes,ID);
			length += a[n]->corr_length;
		}
		length = length / a[i]->n4;
		if(length < _MDR_length1 && score > _MDR_score1){
			_regroup_level += 4;
		} else if(length < _MDR_length2 && score > _MDR_score2){
			_regroup_level += 2;
		}
		if(regroup_level != _regroup_level){
			level1 = _MDR_level1_0 - _MDR_level1_1 * (double)_regroup_level;
			max_cluster = (_MDR_max_cluster_0 + _MDR_max_cluster_1 * (double)_regroup_level) * (double)number_of_genomes;
		} else {
			level1 = level1_bak;
			max_cluster = max_cluster_bak;
		}

		if((n4 = a[i]->n4) > max_cluster) continue;		/* criterion 2 */ /* This is necessary for each cycle. */
		if(nR1 > max_cluster) continue;

		/* creating rel_list */
		for(k=0;k<max_rel;k++){
			rel_list[3 * k] = 0;
			rel_list[3 * k + 1] = 0;
			rel_list[3 * k + 2] = 0;
		}
		max_rel = 0;
		for(j=0;j<nR1;j++){
			ID = a[i]->relatives[j].pID;
			for(n=0;n<max_rel;n++){
				if(rel_list[3 * n] == ID){
					rel_list[3 * n + 1] += 1;
					rel_list[3 * n + 2] += a[i]->relatives[j].count;
					break;
				}
			}
			if(n==max_rel){
				rel_list[3 * n] = ID;
				rel_list[3 * n + 1] = 1;
				rel_list[3 * n + 2] = a[i]->relatives[j].count;
				max_rel += 1;
			}
		}	/* end of creating rel_list */

		/* Too many cross links are not allowed. */
		/* Unacceptable items are also removed. */
		for(j=0;j<max_rel;j++){
			ID = rel_list[3 * j];
			count = rel_list[3 * j + 1];
			cross_edge = (double)rel_list[3 * j + 2];
			n = FindNode(a,nodes,ID);
			n4x = a[n]->n4;

			to_merge = FALSE;

			domain_measure = 0.7 * (double)length / (double)_min_domain_size;
			if(domain_measure > 3.0) domain_measure = 3.0; 
			cross_edge0 = cross_edge_parameter * (sqrt(n4 * n4x) - 0.9) * domain_measure;

			if((n4 >= num_org/2 && n4x < SA) || (n4 < SA && n4x >= num_org/2)) small_addition = TRUE;

			if(cross_edge >= n4 * n4x / 2) to_merge = TRUE;
			else if(cross_edge / cross_edge0 > 3 || cross_edge - cross_edge0 > (double)(Evaluation2((n4 + n4x)/2))) to_merge = TRUE;

			if(!small_addition && (cross_edge <= floor(cross_edge0) + 0.0001)) to_merge = FALSE;

			if(n4x == 0) to_merge = FALSE;
			if(a[n]->domain != 0) to_merge = FALSE;
			if(n4x + n4 > max_cluster) to_merge = FALSE;
			if(n4 == 1 && n4x > 1) to_merge = FALSE;

			/* check for length */
			length2 = 0;
			length3 = 0;
			for(j2=0;j2<n4x;j2++){
				IDx = a[n]->idlist4[j2];
				nn = FindNode(a,nodes,IDx);
				length2 += a[nn]->corr_length;
				length3 += a[nn]->len;
			}
			length2 = length2 / n4x;
			length3 = length3 / n4x;
			if ((double)length3 > (double)length * level2 || (double)length > (double)length3 * level2){
				if ((double)length2 > (double)length * level2 || (double)length > (double)length2 * level2) to_merge = FALSE;
			}

			/* singletons and multidomains are not considered */
			if(!to_merge ||	n4x == 1 || count == 1 || (count == 2 && cross_edge < 3)){
				rel_list[3 * j] = rel_list[3 * (max_rel - 1)];	
				rel_list[3 * j + 1] = rel_list[3 * (max_rel - 1) + 1];	
				rel_list[3 * j + 2] = rel_list[3 * (max_rel - 1) + 2];
				rel_list[3 * (max_rel - 1)] = 0;
				rel_list[3 * (max_rel - 1) + 1] = 0;	
				rel_list[3 * (max_rel - 1) + 2] = 0;
				max_rel--;
				if(max_rel == 0) break;
				j--;
				continue;
			}
		}

		/* Valid rel_list is copied to a[i]->rel_list; */
		if((a[i]->rel_list = (unsigned*)(calloc(3 * max_rel + 3,sizeof(unsigned))))==NULL){
			fprintf(stderr,"Memory allocation error in MergeDistantRelatives 1.\n");
			exit(1);
		}
		for(k=0;k<3 * max_rel;k++){
			a[i]->rel_list[k] = rel_list[k];
		}
		a[i]->max_rel = max_rel;
/*		printf("%u rel_list for %s has been created.\n",i,a[i]->name); 
*/
		printf("%s ",a[i]->name); 
		if(++ct % 10 == 0) printf("\n");
	}
	printf("rel_list finished.\n");

	ct = 0;
	/* Looking for cliques of groups */
	printf("\nMDR stage 2: Searching for cliques of groups ...\n");
	for(i=1;i<=nodes;i++){
		/* rel_list does not include the node itself */
		if((max_rel = a[i]->max_rel) > 0 && max_rel < clique_size){
			to_exit = FALSE;
			for(j=0;j<max_rel;j++){
				n = FindNode(a,nodes,a[i]->rel_list[3*j]);
				if(a[n]->max_rel == max_rel){
					for(k=0;k<a[n]->max_rel;k++){
						if(a[n]->rel_list[3*k] == a[i]->n0){
							break;
						}
					}
					if(k == a[n]->max_rel){
						to_exit = TRUE;
					   	break;
					}
					for(k=0;k<a[n]->max_rel;k++){
						if(a[n]->rel_list[3*k] == a[i]->n0) continue;	
						for(j2=0;j2<max_rel;j2++){
							if(a[i]->rel_list[3*j2] == a[n]->n0) continue;
							if(a[n]->rel_list[3*k] == a[i]->rel_list[3*j2])	break;
						}
						if(j2==max_rel){
							to_exit = TRUE;
							break;
						}
					}
					if(to_exit) break;
				} else {
					to_exit = TRUE;
					break;
				}
			}
			if(to_exit) continue;

			/* check if resultant n4 is too large. */
			n4_total = a[i]->n4;
			for(j=0;j<=max_rel;j++){
				n = FindNode(a,nodes,a[i]->rel_list[3*j]);
				n4_total += a[n]->n4;
			}
			if(n4_total > max_cluster + (unsigned)(_MDR_max_cluster_1 * (double)clique_size * (double)number_of_genomes)) continue;

			/* cliques are marked by negative value of max_rel */
			a[i]->max_rel = -max_rel;
			for(j=0;j<=max_rel;j++){
				n = FindNode(a,nodes,a[i]->rel_list[3*j]);
				a[n]->max_rel = -max_rel;
			}
			printf("%s ",a[i]->name);
			if(++ct % 10 == 0) printf("\n");
		}
	}

	
	/* Start of merging cliques of groups */
	printf("\nMDR stage 3: Merging cliques of groups ...\n");
	for(i=1;i<=nodes;i++){
		if(a[i]->max_rel < 0) max_rel = -(a[i]->max_rel);
		else continue;

		nR1=a[i]->nR;
		n4 = a[i]->n4;

		for(j=0;j<max_rel;j++){
			n = FindNode(a,nodes,a[i]->rel_list[3 * j]);
			n4x = a[n]->n4;

			/* integrating a[n] into a[i] */
			if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n4 + n4x + 1)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in MergeDistantRelatives2.\n");
				exit(1);
			}
			if(j==0){
				if(max_rel == 1) strcpy(word,"group");
				else strcpy(word,"groups");
				printf("Merging cliques of %d %s: group %u %s n4=%u to group %u %s n4=%u\n",max_rel,word,a[n]->n0,a[n]->name,a[n]->n4,\
					a[i]->n0,a[i]->name,a[i]->n4);
			} else {
				printf("          : group %u %s n4=%u to group %u %s n4=%u\n",a[n]->n0,a[n]->name,a[n]->n4,\
					a[i]->n0,a[i]->name,a[i]->n4);
			}
			for(k=0;k<n4x;k++){
				m = a[i]->idlist4[n4+k] = a[n]->idlist4[k];
				mx = FindNode(a,nodes,m);
				a[mx]->pID = a[i]->n0;
			}
			n4 += n4x;
			a[i]->n4 = n4;
			a[n]->n4b = a[n]->n4;
			a[n]->n4 = 0;
			a[n]->n3b = a[n]->n3;
			a[n]->n3 = 0;

			for(k=j;k<nR1;k++){
				if(a[i]->relatives[k].pID == ID){
					a[i]->relatives[k].ID = a[i]->relatives[nR1-1].ID;
					a[i]->relatives[k].pID = a[i]->relatives[nR1-1].pID;
					a[i]->relatives[k].grpno = a[i]->relatives[nR1-1].grpno;
					a[i]->relatives[k].count = a[i]->relatives[nR1-1].count;
					nR1--;
					k--;
					if(nR1 == 0) break;
				}
			}
			a[i]->nR = nR1;
			a[n]->max_rel = 0;
		}
		a[i]->max_rel = 0;
	}


	/* Resetting the list */
	for(k=0;k<max_nR1;k++){
		rel_list[3 * k] = 0;
		rel_list[3 * k + 1] = 0;
		rel_list[3 * k + 2] = 0;
	}
	max_rel = 0;

	/* Start of another round of merging */
	printf("\nMDR stage 4: Merging other groups ...\n");
	for(i=1;i<=nodes;i++){
		if(a[i]->n3 == 0 || a[i]->n4 == 0) continue;
		if(a[i]->domain != 0) continue;		/* criterion 1 */
		nR1=a[i]->nR;
		n4 = a[i]->n4;

		/* 355 */
		_regroup_level = regroup_level;

		/* score of self similarity */
		score = thr_list[0];
		for(j=0;j<a[i]->n3;j++){
			if(a[i]->sqlist3[j].ID == a[i]->n0 && a[i]->sqlist3[j].qID == a[i]->n0){
				if(score > a[i]->sqlist3[j].score) score = a[i]->sqlist3[j].score;
			}
		}
		/* average corrected length */
		length = 0;
		for(j=0;j<a[i]->n4;j++){
			ID = a[i]->idlist4[j];
			n = FindNode(a,nodes,ID);
			length += a[n]->corr_length;
		}
		length = length / a[i]->n4;
/*
printf("score[0]=%e.\n",score);
*/
		if(length < _MDR_length1 && score > _MDR_score1){
			_regroup_level += 4;
		} else if(length < _MDR_length2 && score > _MDR_score2){
			_regroup_level += 2;
		}
		if(regroup_level != _regroup_level){
			level1 = _MDR_level1_0 - _MDR_level1_1 * (double)_regroup_level;
			max_cluster = (_MDR_max_cluster_0 + _MDR_max_cluster_1 * (double)_regroup_level) * (double)number_of_genomes;
		} else {
			level1 = level1_bak;
			max_cluster = max_cluster_bak;
		}
		/* end 355 */
			
		if(n4 > max_cluster) continue;		/* criterion 2 */ /* This is necessary for each cycle. */
		if(nR1 > max_cluster) continue;

		/* creating rel_list */
		for(k=0;k<max_rel;k++){
			rel_list[3 * k] = 0;
			rel_list[3 * k + 1] = 0;
			rel_list[3 * k + 2] = 0;
		}

		max_rel = 0;
		for(j=0;j<nR1;j++){
			ID = a[i]->relatives[j].pID;
			for(n=0;n<max_rel;n++){
				if(rel_list[3 * n] == ID){
					rel_list[3 * n + 1] += 1;
					rel_list[3 * n + 2] += a[i]->relatives[j].count;
					break;
				}
			}
			if(n==max_rel){
				rel_list[3 * n] = ID;
				rel_list[3 * n + 1] = 1;
				rel_list[3 * n + 2] = a[i]->relatives[j].count;
				max_rel += 1;
			}
		}	/* end of creating rel_list */

		for(j=0;j<max_rel;j++){
			ID = rel_list[3 * j];
			count = rel_list[3 * j + 1];
			cross_edge = (double)rel_list[3 * j + 2];
			n = FindNode(a,nodes,ID);
			/* the following two lines are not used */
			if((n4x = a[n]->n4) == 0) continue;
			if(a[n]->domain != 0) continue;		/* criterion 1 */
			if(n4x + n4 > max_cluster) continue;		/* criterion 2 */
			if(n4 == 1 && n4x > 1) continue;

			to_merge = 0;

			/* 355n */
			domain_measure = 0.7 * (double)length / (double)_min_domain_size;
			if(domain_measure > 3.0) domain_measure = 3.0; 
			cross_edge0 = cross_edge_parameter * (sqrt(n4 * n4x) - 0.9) * domain_measure;

			if((n4 >= num_org/2 && n4x < SA) || (n4 < SA && n4x >= num_org/2)) small_addition = TRUE;

			if(!small_addition && (cross_edge <= floor(cross_edge0) + 0.0001)) continue;

			/* check for length */
			length2 = 0;
			length3 = 0;
			for(j2=0;j2<n4x;j2++){
				IDx = a[n]->idlist4[j2];
				nn = FindNode(a,nodes,IDx);
				length2 += a[nn]->corr_length;
				length3 += a[nn]->len;
			}
			length2 = length2 / n4x;
			length3 = length3 / n4x;
			if ((double)length3 > (double)length * level2 || (double)length > (double)length3 * level2){
				if ((double)length2 > (double)length * level2 || (double)length > (double)length2 * level2) continue;
			}

			if(cross_edge >= n4 * n4x / 2) to_merge = 5;
			if(cross_edge / cross_edge0 > 3 || cross_edge - cross_edge0 > (double)(Evaluation2((n4 + n4x)/2))) to_merge = 3;

			if(length > _MDR_length2 && score < _MDR_score2){
				if(!DeltaTest(a,nodes,i,n)) continue;
			}

			/* criterion 4 */	
			/* 355 n4x should be limited */
			if((double)count >= level1 * (double)n4x) to_merge = 1;
			/* criterion 5 */
			else if(count + 1 >= nR1){	/* 353g */
				nR2 = a[n]->nR;
				count2 = 0;
				for(j2=0;j2<nR2;j2++){
					if(a[n]->relatives[j2].pID == a[i]->n0) count2 += 1;
				}
				if(count2 == nR2) to_merge = 2;	/* 353g */
			}

			/* criterion 3 */
			/* A modified value of _regroup_level is used in CountOrgInIDlist */
			switch(CountOrgInIDlist(a,nodes,i,n)){
				case 0:
					to_merge = 0;	
					break;
				case -1:
					to_merge = -1;	/* sign of error */
					break;
				case 1: 
					/*
					if(to_merge == 1 || to_merge == 2) break;
					else {
					*/
						to_merge = 0;
						break;
						/*
					}*/
				case 2: 
					if(to_merge == 1 || to_merge == 2 || to_merge == 3 || to_merge == 5) break;
					else {
						to_merge = 0;
						break;
					}
				case 3: 
					to_merge = 4;
					break;
				default:
					to_merge = 0;
					break;
			}
				
			if(to_merge > 0){

				/* integrating a[n] into a[i] */
				if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n4 + n4x + 1)*sizeof(unsigned)))==NULL){
					fprintf(stderr,"Memory allocation error in MergeDistantRelatives3.\n");
					exit(1);
				}
				/* 355 */
				printf("Merging group %u %s n4=%u to group %u %s n4=%u, to_merge=%d",a[n]->n0,a[n]->name,a[n]->n4,\
						a[i]->n0,a[i]->name,a[i]->n4,to_merge);
				if(regroup_level != _regroup_level){
					printf(", with regroup level %d for %s.\n",_regroup_level,a[i]->name);
				} else {
					printf(".\n");
				}

				for(k=0;k<n4x;k++){
					m = a[i]->idlist4[n4+k] = a[n]->idlist4[k];
					mx = FindNode(a,nodes,m);
					a[mx]->pID = a[i]->n0;
				}
				n4 += n4x;
				a[i]->n4 = n4;
				a[n]->n4b = a[n]->n4;
				a[n]->n4 = 0;
				a[n]->n3b = a[n]->n3;
				a[n]->n3 = 0;

				a[n]->max_rel = 0;

/*Note:  	 	ID = a[i]->relatives[j].pID;
				n = FindNode(a,nodes,ID);
			*/
/* code corrected in 355 
 * k=j+1 -> k=j, relatives[nR1] -> relatives[nR1-1] */

				for(k=j;k<nR1;k++){
					if(a[i]->relatives[k].pID == ID){
						a[i]->relatives[k].ID = a[i]->relatives[nR1-1].ID;
						a[i]->relatives[k].pID = a[i]->relatives[nR1-1].pID;
						a[i]->relatives[k].grpno = a[i]->relatives[nR1-1].grpno;
						a[i]->relatives[k].count = a[i]->relatives[nR1-1].count;
						nR1--;
						k--;
						if(nR1 == 0) break;
					}
				}
				a[i]->nR = nR1;
			}	/* end to_merge */

		}	/* j */
	}	/* i */
	_regroup_level = regroup_level;

}
/*********************************************************/

/*********************************************************/
Node **AddList2(Node **a,unsigned nodes,FILE *missing)
/* Here, sqlist2 is constructed by recurrently indexing
a[i]->list1. 
Simplified indexing. Dec. 24, 2003.
Now n2b is used as the allocation size, while n2 indicates
real size of sqlist2.*/
/*********************************************************/
{
#define STEPSIZE 10
	unsigned i,j;
	unsigned k,m,n2;
	unsigned A;

	for(i=1;i<=nodes;i++){
		A=a[i]->n0;
		if(A != i) fprintf(stderr,"Number strange. i=%u, A=%u.\n",i,A);
		a[i]->n2b = a[i]->n1 + 1;	/* just an approximation */
		if((a[i]->sqlist2=(SQlist*)calloc(a[i]->n2b,sizeof(SQlist)))==NULL){
			fprintf(stderr, "Memory allocation error in AddList2-1\n.");
			return NULL;
		}
	}

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(a[i]->n1==0) continue;
		if(a[i]->n1==1){
			RcopySQlist(&a[i]->sqlist2[0],&a[i]->sqlist1[0]);
			a[i]->n2 = 1;
			continue;
		}
		for(k=0;k<a[i]->n1;k++){
			if(a[i]->sqlist1[k].score >= thr * 0.999) continue;
			if(a[i]->sqlist1[k].Sstart==0 && a[i]->sqlist1[k].Send==0) continue;
			if(a[i]->sqlist1[k].qID != i){
				fprintf(stderr,"qID is not correct. i=%u, k=%u, qID=%u.\n",i,k,a[i]->sqlist1[k].qID);
				continue;
			}
			j=a[i]->sqlist1[k].ID;
			if(j==0 || j > nodes) continue;

			n2 = a[j]->n2;
			if(a[j]->n2b <= a[j]->n2 + 1){
				a[j]->n2b = n2 + STEPSIZE;
				if((a[j]->sqlist2=(SQlist*)realloc(a[j]->sqlist2,(n2+STEPSIZE+1)*sizeof(SQlist)))==NULL){
					fprintf(stderr, "Memory allocation error in AddList2-2.\n");
					fprintf(stderr, "i=%u, j=%u, k=%u, a[j]->n2=%u, a[j]->n2b=%u\n.",i,j,k,a[j]->n2,a[j]->n2b);
					return NULL;
				}
				for(m=0;m<STEPSIZE;m++) clearSQlist(&a[j]->sqlist2[n2+m]);
			}

			RcopySQlist(&a[j]->sqlist2[n2],&a[i]->sqlist1[k]);
			a[j]->n2 += 1;
		}
	}

	for(i=1;i<=nodes;i++){
		a[i]->n2b = 0;

		if(a[i]->n2 == 0 && a[i]->n1 == 0){
			fprintf(missing,"\nHomology data is missing for i=%u: %s.\n",i,a[i]->name);
			fflush(missing);
		}
	}

	return a;
}
/*********************************************************/

/*********************************************/
void clearSQlist(SQlist *A)
/* This function writes 0 to SQlistA. */
/*********************************************/
{
	if(A==NULL) return;

	A->ID = 0;
	A->Sstart = 0;
	A->Send = 0;
	A->score = 1.0;
	A->qID = 0;
	A->Qstart = 0;
	A->Qend = 0;
	A->domain = 0;
	A->crosslink = FALSE;

	return;
}
/*********************************************/

/*********************************************/
void copySQlist(SQlist *A, SQlist *B)
/* This function copies SQlistB to SQlistA. */
/*********************************************/
{
	if(A==NULL || B==NULL) return;

	A->ID = B->ID;
	A->Sstart = B->Sstart;
	A->Send = B->Send;
	A->score = B->score;
	A->qID = B->qID;
	A->Qstart = B->Qstart;
	A->Qend = B->Qend;
	A->domain = B->domain;
	A->crosslink = B->crosslink;
	return;
}
/*********************************************/

/*********************************************/
void RcopySQlist(SQlist *A, SQlist *B)
/* This function copies SQlistB to SQlistA,
but exchanging Q and S. */
/*********************************************/
{
	if(A==NULL || B==NULL) return;

	A->ID = B->qID;
	A->Sstart = B->Qstart;
	A->Send = B->Qend;
	A->score = B->score;
	A->qID = B->ID;
	A->Qstart = B->Sstart;
	A->Qend = B->Send;
	A->domain = B->domain;
	A->crosslink = B->crosslink;

	return;
}
/*********************************************/

/****************************************************************/
int cmpSQlist(SQlist *A, SQlist *B, double level1, double level2)
/* This function compares SQlistA and SQlistB,
and returns 1 if the two are judged to be 
identical. This means that the ID should be 
exactly identical, but the regions may be 
slightly different. The score is not examined.
If A contains B, this function returns 2.
If B contains A, this function returns 3. 
level1 is used to judge identity. minimal overlap. 
level2 is used to judge implication. fold 

Division by zero and initial state data is avoided, although this 
should not happen in normal data set.
 */
/****************************************************************/
{
	unsigned ASlength,AQlength,BSlength,BQlength;
	unsigned Sover,Qover;
	double kS,kQ;
	double k2S,k2Q;

	if(A->ID != B->ID) return 0;
	if(A->qID != B->qID) return 0;

	if(A->Send == A->Sstart || A->Sstart == 0 || A->Send == 0) return 0;
	else if(A->Send > A->Sstart) ASlength = A->Send - A->Sstart + 1;
	else ASlength=A->Sstart - A->Send + 1;
	if(A->Qend == A->Qstart || A->Qstart == 0 || A->Qend == 0) return 0;
	else if(A->Qend > A->Qstart) AQlength=A->Qend - A->Qstart + 1;
	else AQlength=A->Qstart - A->Qend + 1;
	if(B->Send == B->Sstart || B->Sstart == 0 || B->Send == 0) return 0;
	else if(B->Send > B->Sstart) BSlength=B->Send - B->Sstart + 1;
	else BSlength=B->Sstart - B->Send + 1;
	if(B->Qend == B->Qstart || B->Qstart == 0 || B->Qend == 0) return 0;
	else if(B->Qend > B->Qstart) BQlength=B->Qend - B->Qstart + 1;
	else BQlength=B->Qstart - B->Qend + 1;

	Sover=overlap(A->Sstart,A->Send,B->Sstart,B->Send);
	Qover=overlap(A->Qstart,A->Qend,B->Qstart,B->Qend);

	if(ASlength > BSlength) {
		kS=(double)Sover/(double)BSlength;
		k2S=(double)ASlength/(double)BSlength;
	} else {
		kS=(double)Sover/(double)ASlength;
		k2S=(double)BSlength/(double)ASlength;
	}
	if(AQlength > BQlength) {
		kQ=(double)Qover/(double)BQlength;
		k2Q=(double)AQlength/(double)BQlength;
	} else {
		kQ=(double)Qover/(double)AQlength;
		k2Q=(double)BQlength/(double)AQlength;
	}

	if(kS>level1 || kQ>level1){
		if(k2S > level2 || k2Q > level2){
			if(ASlength > BSlength) return 2;
			else return 3;
		}
		else return 1;
	}

	return 0;
}
/****************************************************************/


/*********************************************************/
void clearN3list(Node **a,unsigned nodes)
/*********************************************************/
{
	unsigned i,j;
	unsigned len3;

	for(i=1;i<=nodes;i++){
/*		if(a[i]->n3b == 0) continue;
*/
		if(!a[i]->active) continue;

		len3 = a[i]->len3;
		for(j=0;j<len3;j++){
			clearSQlist(&a[i]->sqlist3[j]);
		}
/*		if(a[i]->final_thr != 1.0) continue;
*/
		a[i]->n3=0;
		a[i]->n3b=0;
	}
	return;
}
/*********************************************************/



/*********************************************************/
void SearchBridge(Node **a,unsigned nodes)

/* New in version 308.
This subroutine searches bridging nodes, which show homology
to two different groups of proteins. */ 

/* domain ID is expressed as 1,2,4,8,16,32,64,128, or
0x01, 0x02, 0x04, 0x08, 0x10, 0x11, 0x12, 0x14, 0x18.
These represent ID numbers, 1,2,3,4,5,6,7,and 8.
If 8 different domains are present, 0xFF. Maximum number of
domains is limited to 30.*/
/* DMlist is initalized in main(). DMnum = 1 before start. */
/* DMmax is used to hold the maximum number of domain ID.
   Homologous domains within a protein is given an
   identical domain ID. */
/*********************************************************/
{
	unsigned i,j,m,jj,k1;
	unsigned n3,n4;
	unsigned a1,a2,b1,b2,x,max_n4;
	double score;
	double level1 = _level1 * 0.9;
	double level3 = _level3;
	unsigned min_domain_size = _min_domain_size * 1.2;
	double searchbridge_min_triangle = _searchbridge_min_triangle;
	double searchbridge_min_triangle2 = _searchbridge_min_triangle / 2;
	unsigned searchbridge_min_n4;
	DMlr *dmlr_1;
	unsigned DMmax;
	unsigned long *dmID;
	unsigned long maskID=0x00;
	unsigned long tmpID=0x00;
	unsigned long tmpID2=0x00;
	unsigned total,counter,counter2,counter01,threshold,triangle;


	total=counter=threshold=0;
	if(_searchbridge_min_n4 == 0) searchbridge_min_n4 = number_of_genomes;
	else searchbridge_min_n4 = _searchbridge_min_n4;

/* Defining domains */
	printf("Defining domains...\n");

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(a[i]->domain == 0x4) continue;
		n3=a[i]->n3;
		if(n3 < 2) continue;
		a1=a2=b1=b2=0;

		/* assignment of domains. for each of idlist3. */
		for(jj=0;jj<n3;jj++){
			if(a[i]->sqlist3[jj].qID == i){
				b1=a[i]->sqlist3[jj].Qstart;
				b2=a[i]->sqlist3[jj].Qend;
			} else {
				continue;
			}
			for(m=0;m<a[i]->DMnum;m++){
				a1=a[i]->dmlist[m].start;
				a2=a[i]->dmlist[m].end;
				score = OverlapScore(a1,a2,b1,b2);
				if(score > level1) break;
				else if(level3 < score && score <= level1){ 
					dmlr_1 = OverlapDomains(a1,a2,b1,b2);
					if(dmlr_1->overlap <= min_domain_size) continue;

					if(dmlr_1->Al > min_domain_size){
						a[i]->dmlist=expandDMlist(a[i]->dmlist,a[i]->DMnum++,m);
						if(error_expandDMlist){
							a[i]->DMnum--;
							printf("i=%u, %s.\n",i, a[i]->name);
							PrintDomain(stdout,a,nodes,i);
							exit(1);
						}
						a[i]->dmlist[m+1].start=b1;
						a[i]->dmlist[m].end=b1-1;
						DMmax = a[i]->DMmax++;
						a[i]->dmlist[m+1].ID = bit2[DMmax+1];
						m++;
					}
					if(dmlr_1->Ar > min_domain_size){
						a[i]->dmlist=expandDMlist(a[i]->dmlist,a[i]->DMnum++,m);
						if(error_expandDMlist){
							a[i]->DMnum--;
							printf("i=%u, %s.\n",i, a[i]->name);
							PrintDomain(stdout,a,nodes,i);
							exit(1);
						}
						a[i]->dmlist[m+1].start=b2+1;
						a[i]->dmlist[m].end=b2;
						DMmax = a[i]->DMmax++;
						a[i]->dmlist[m+1].ID = bit2[DMmax+1];
						m++;
					}
					if(a2 < b2) b1 = a2 + 1;	/* for the next cycle */
					break;
				}
			}		/* m */
		}		/* jj */
	} /* i */

	if(verbous_mode){
		printf("\n");	
		PrintDomain2(stdout,a,nodes);
		fflush(stdout);
	}

/* Counting effective number of domains. */

	max_n4 = 0;
	for(i=1;i<=nodes;i++){
		if(max_n4 < a[i]->n4) max_n4 = a[i]->n4;
	}
	if((dmID = (unsigned long*)calloc(max_n4 + 2,sizeof(unsigned long)))==NULL){
		fprintf(stderr,"Memory allocation error in SearchBridge 1.\n");
		exit(1);
	}

	for(i=1;i<=nodes;i++){
		n3 = a[i]->n3;
		n4 = a[i]->n4;
		if(a[i]->DMnum == 1 || n3 < 2 || n4 < 2){
			a[i]->effDMnum = 0;
			continue;
		} else if(n3 < 3 || n4 < 3){	/* if only a single homolog other than the query itself is found */
			a[i]->effDMnum = 1;
			continue;
		}

		for(j=0;j<n4;j++){
			if((k1 = a[i]->idlist4[j]) == i) continue;
			dmID[j] = 0x00;
			for(jj=0;jj<n3;jj++){
				if(a[i]->sqlist3[jj].ID == k1){
					b1=a[i]->sqlist3[jj].Qstart;
					b2=a[i]->sqlist3[jj].Qend;
					for(m=0;m<a[i]->DMnum;m++){
						a1=a[i]->dmlist[m].start;
						a2=a[i]->dmlist[m].end;
						x = overlap(a1,a2,b1,b2);
						if(((double)x / (double)(a2 - a1)) >= level1){
							dmID[j] |= a[i]->dmlist[m].ID;
						}
					}
				}
			}
		}
		maskID = 0x00;
		for(m=0;m<a[i]->DMnum;m++){
			if((a[i]->dmlist[m].end - a[i]->dmlist[m].start) < min_domain_size) continue;
			maskID |= a[i]->dmlist[m].ID;
		}
		total = CountBits(maskID);
		a[i]->effDMnum = total;
		if(total < 2) continue;
		threshold = (unsigned)((total + 4) / 3);
			/* This part is essential in judgeing MD.
			 * total is the number of domains used for this judgement, 
			 * which are larger than 2 times min_domain_size.
			 * threshold is the minimum number of domains, 
			 * which are shared by only one of the pair of proteins to be compared,
			 * namely, the two proteins are said to be unrelated.
			 * example: total  2 3 4 5 6 7
			        threshold  2 2 2 3 3 3 
			*/

/* for debug 
	printf("Node %d, maskID= %ld, total= %d, threshold= %d. \n",i,maskID,total,threshold);
*/
		a[i]->maskID = maskID;

		if(n4 < searchbridge_min_n4) continue;
		counter = 0;
		counter2 = 0;
		counter01 = 0;
		for(j=1;j<n4-1;j++){
			for(jj=j+1;jj<n4;jj++){
				tmpID2 = (dmID[j] & dmID[jj]) & maskID;	/* counting completely matching IDs */
				if(CountBits(tmpID2) == total) counter2 += 1;

				tmpID = (dmID[j] ^ dmID[jj]) & maskID;	/* counting incompatible IDs */
				if(CountBits(tmpID) >= threshold) counter += 1;	
				
				/* counting really incompatible IDs, namely, 1-0 in a certain bit and 0-1 in another bit. */
				tmpID = ((dmID[j] & maskID) & (dmID[jj] ^ maskID));
				tmpID2 = ((dmID[j] ^ maskID) & (dmID[jj] & maskID));
				if((CountBits(tmpID) > 0) && (CountBits(tmpID2) > 0)) counter01 += 1;
			}	
		}
/* for debug 
	printf("Node %d, counter= %d, n4= %d.\n",i,counter,n4);
	fflush(stdout);
*/
		triangle = (n4 - 1) * (n4 - 2) / 2;

		/* Finally, the protein i is judged to be MD,
		 * if more than a half (or a proportion as indicated by searchbridge_min_triangle)
		 * of protein pairs are found unrelated (as defined above).
		 * The default value of searchbridge_min_triangle is 0.4.
		 */
		/* 351a4
		 * Complete match is now counted. If domains are completely matched
		 * in more than a certain fraction, the protein i is not regarded 
		 * as a multidomain protein. The level is defined by 
		 * searchbridge_min_triangle2, which is a half of searchbridge_min_triangle.
		 */

		if(counter01 <= n4) continue;
		else if(counter < triangle * searchbridge_min_triangle) continue;
		else if(counter2 > triangle * searchbridge_min_triangle2 || \
				counter2 > number_of_genomes * level1) continue;
		else {
			a[i]->domain = 1;
			printf("Node %d, maskID= %ld, total= %d, threshold= %d. \n",i,maskID,total,threshold);
			printf("Node %d, counter= %d, counter01= %d, n4= %d.\n",i,counter,counter01,n4);
			printf("Node %d %s is being set as a multidomain protein.\n",i,a[i]->name);
			fflush(stdout);
		}

	}

	return;
}
/*********************************************************/


/*********************************************************/
Node **CleanList(Node **a,unsigned nodes)

/* In this subroutine, redundancy in sqlist1 is removed.
Next, the size of sqlist2 is corrected.  Then, sqlist3
is created by copying sqlist2, and incorporating sqlist1.
n30 is the size of sqlist3 at this stage. n3 will be
changed during the subsequent processing. The allocated
memory size of sqlist3 is n1+n2-1. */
/* Now the removed items in the sqlist1 and sqlist2 are
not cleared. The original n1 and n2 are stored as n1b and
n2b, respectively, and the original sqlist1 and sqlist2 
may be made available later. */
/* Now, multidomain is excluded */
/*********************************************************/
{
	unsigned i,j,k1,k2;
	int c;
	unsigned last;
	unsigned m;
	unsigned len3;
	
	if(a==NULL || nodes == 0) return NULL;
	
	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(!a[i]->active){
			if(a[i]->n3 != 0) continue;		/* 308 */
		} else if(a[i]->final_thr < thr || (a[i]->domain != 0)){
			a[i]->n1 = 0;
			a[i]->n2 = 0;
			a[i]->n3 = 0;
			a[i]->n3b = 0;
			continue;
		}
		if(a[i]->sqlist1 == NULL){ 
			fprintf(stderr,"Empty list 1 with i=%u. Skipping.\n",i);
			fflush(stderr);
		} else {

	/* Here, a[i]->sqlist[0] is not modified. */
		    if(a[i]->n1 > 1) {
				for(j=1;j<a[i]->n1 && a[i]->n1>1;j++){
					last=a[i]->n1-1;
					m=FindNode(a,nodes,a[i]->sqlist1[j].ID);
					if(m==i || m==0) continue;
					if(a[m]->final_thr < thr || a[i]->sqlist1[j].score >= thr * 0.999 || \
						(a[m]->domain != 0)) {
						if(j<last){
							swapSQlist(&a[i]->sqlist1[j],&a[i]->sqlist1[last]);
						}
						a[i]->n1 -= 1;
						if(j==last) break;
						else j--;
					}
				}
		    }
		}

		if(a[i]->sqlist2 == NULL){ 
			fprintf(stderr,"Empty list 2 with i=%u. Skipping.\n",i);
			fflush(stderr);
		} else {
			if(a[i]->n2 > 1){
				while(a[i]->n2>1){
					j=a[i]->n2-1;
					if(a[i]->sqlist2[j].ID==0) a[i]->n2 -= 1;
					else break;
				}
			}
			for(j=0;j<a[i]->n2;j++){
				if(a[i]->n2 == 0) break;
				m=FindNode(a,nodes,a[i]->sqlist2[j].ID);
				if(m==i || m==0) continue;
				last=a[i]->n2-1;
				if(a[m]->final_thr < thr || a[i]->sqlist2[j].score >= thr * 0.999 || \
					(a[m]->domain != 0)) {
					if(j<last){
						swapSQlist(&a[i]->sqlist2[j],&a[i]->sqlist2[last]);
					}
					a[i]->n2 -= 1;
					if(j==last) break;
					else j--;
				}
			}
		}

		len3 = a[i]->n1 + a[i]->n2;
		if(len3 < 2) len3 = 1;
		else if(len3 > _max_n3){
			/* bug fix. 353e, Feb. 17, 2006. */
			fprintf(stderr,"len3=%u exceedes max_n3=%u for node=%u %s.\n",len3,_max_n3,a[i]->n0,a[i]->name);
			fflush(stderr);
			if(a[i]->n2 > _max_n3) len3 = a[i]->n2 + 100;
			else len3 = _max_n3;
		}

/* To avoid repeated memory allocation for i=1 from the iteration 2 */
		if(repeat_mode && thr > thr_list[0] && i==1 && (len3 < (unsigned)(nodes * 0.25))){
			len3 = (unsigned)(nodes * 0.25);
			printf("len3 for i=%u is set to %u.\n",i,len3);
		}

		if(a[i]->len3 == 0){
			a[i]->len3 = len3;
			if((a[i]->sqlist3=(SQlist*)calloc(a[i]->len3,sizeof(SQlist)))==NULL){
				fprintf(stderr, "Memory allocation error in CleanList with i=%u\n.",i);
				return NULL;
			}
		} else if(len3 > a[i]->len3){
			if((a[i]->sqlist3=(SQlist*)realloc(a[i]->sqlist3,len3*sizeof(SQlist)))==NULL){
				fprintf(stderr, "Memory allocation error in CleanList with i=%u\n.",i);
				return NULL;
			}
			a[i]->len3 = len3;
		}

		a[i]->n3 = 0;
		a[i]->n3b = a[i]->n3;
		
		if(a[i]->sqlist1 == NULL && a[i]->sqlist2 != NULL){
			for(j=0;j<a[i]->n2;j++){
				copySQlist(&a[i]->sqlist3[j],&a[i]->sqlist2[j]);
			}
			a[i]->n3 = a[i]->n2;
			a[i]->n3b = a[i]->n3;
			a[i]->n30 = a[i]->n2;
			continue;
		}

		if(a[i]->sqlist1 != NULL && a[i]->sqlist2 == NULL){
			for(j=0;j<a[i]->n1;j++){
				copySQlist(&a[i]->sqlist3[j],&a[i]->sqlist1[j]);
			}
			a[i]->n3 = a[i]->n1;
			a[i]->n3b = a[i]->n3;
			a[i]->n30 = a[i]->n1;
			continue;
		}
			
		if(a[i]->sqlist1 == NULL && a[i]->sqlist2 == NULL){
			a[i]->n3 = 0;
			a[i]->n3b = a[i]->n3;
			a[i]->n30 = 0;
			continue;
		}


/* By default, SQlist3 is first formed as a copy of SQlist2. */
		for(j=0;j<a[i]->n2;j++){
			copySQlist(&a[i]->sqlist3[j],&a[i]->sqlist2[j]);
		}

/* Adding SQlist1. */
/* j=a[i]->n2 at this stage. */
		for(k1=0;k1<a[i]->n1;k1++){
			for(k2=0;k2<j;k2++){
				c=cmpSQlist(&a[i]->sqlist1[k1],&a[i]->sqlist3[k2],_level1,_level2);
				if(c==1 || c==3) break;	/* sqlist1 is identical to or included by sqlist2 */
				else if(c==2){	/* sqlist1 includes sqlist2 */
					copySQlist(&a[i]->sqlist3[k2],&a[i]->sqlist1[k1]);
					if(len3 <= k2 + 2){
						if(len3 > _max_n3 -10) fprintf(stderr,"len3=%u, k2=%u\n",len3,k2);
						fflush(stderr);
				/*		len3 += 10; */
						len3 = k2 + 10;
						if((a[i]->sqlist3=(SQlist*)realloc(a[i]->sqlist3,len3*sizeof(SQlist)))==NULL){
							fprintf(stderr, "Memory allocation error in CleanList 3 with i=%u\n.",i);
							exit(1);
						}
						a[i]->len3 = len3;
					}
					break;
				} 
			}
			if(k2==j){
				if(a[i]->sqlist1[k1].ID > nodes || a[i]->sqlist1[k1].ID==0) continue;
				if(len3 <= k2 + 2){
			/*		len3 += 10;	*/
					if(len3 > _max_n3 -10) fprintf(stderr,"len3=%u, k2=%u\n",len3,k2);
					fflush(stderr);
					len3 = k2 + 10;
					if((a[i]->sqlist3=(SQlist*)realloc(a[i]->sqlist3,len3*sizeof(SQlist)))==NULL){
						fprintf(stderr, "Memory allocation error in CleanList 4 with i=%u\n.",i);
						exit(1);
					}
					a[i]->len3 = len3;
				}
				copySQlist(&a[i]->sqlist3[j++],&a[i]->sqlist1[k1]);
			}
		}
		a[i]->n3 = j;
		a[i]->n3b = a[i]->n3;

/* At this stage, a[i]->sqlist3[j] (j>=a[i]->n3) remain initialized. */

/* counting non-redundant IDs and set it in n30. No change in sqlist3. */
		m=1;
		for(k1=1;k1<j;k1++){
			for(k2=0;k2<k1;k2++){
				if(a[i]->sqlist3[k2].ID == a[i]->sqlist3[k1].ID) break;
			}
			if(k2==k1) m+=1;
		}
		a[i]->n30 = m;
	}
	return a;
}
/*********************************************************/



/******************************************************************/
Node **BinaryInteractions(Node **a,unsigned nodes)

/* This function creates matchlist7, which contains information
on best score, overlap score, and dmID for each pair of nodes 
that are present in sqlist3 as directly read from data.out.
This is used in clique mode from version 3.50. */
/* Version 355. Transit peptide is explicitly considered in 
estimating eukaryote-prokaryote relations. */ 
/******************************************************************/
{
	unsigned i,j,k,m,t1,t2;
	unsigned n3,n4,n7;
	unsigned a1,a2,b1,b2;
	unsigned ID;
	unsigned x;
	double level1 = _level1 * 0.9;
	unsigned corr_length1,corr_length2;
	unsigned start1;
	double score;
	double overlap_value;
	unsigned long dmID;
	SQlist *t;

	t1 = GetMax_n3(a,nodes);
	if((t=(SQlist*)calloc(t1,sizeof(SQlist)))==NULL){
		fprintf(stderr,"Error in BinaryInteractions0-1.\n");
		exit(1);
	}

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(a[i]->domain & 0x04) continue;
/*		if(a[i]->domain != 0 || !a[i]->active) continue;
355 Matchlist7 is made for multidomain proteins */
		n3= a[i]->n3;
		n4= a[i]->n4;
		n7 = a[i]->n7 = n4;
	
		if((a[i]->matchlist7=(Match*)calloc(n7+1,sizeof(Match)))==NULL){
			fprintf(stderr, "Memory allocation error in BinaryInteractions1 with i=%u\n.",i);
			return NULL;
		}

/*
printf("i=%u, DMnum=%u.\n",i,a[i]->DMnum);
*/

		for(j=0;j<n4;j++){
			
			ID = a[i]->matchlist7[j].ID = a[i]->idlist4[j];

			score = 1.0;
			for(k=0;k<n3;k++){
				if(ID == a[i]->sqlist3[k].ID && \
					a[i]->sqlist3[k].qID == a[i]->n0){
					if(score > a[i]->sqlist3[k].score) score = a[i]->sqlist3[k].score;
				}
			}
			a[i]->matchlist7[j].score = score;
	
			dmID = 0x00;
			for(k=0;k<n3;k++){
				if(ID == a[i]->sqlist3[k].ID && a[i]->sqlist3[k].qID == a[i]->n0){
					b1=a[i]->sqlist3[k].Qstart;
					b2=a[i]->sqlist3[k].Qend;
					for(m=0;m<a[i]->DMnum;m++){
						a1=a[i]->dmlist[m].start;
						a2=a[i]->dmlist[m].end;
						x = overlap(a1,a2,b1,b2);

/*
printf("DMnum=%u,x=%u, a1=%u, a2=%u, b1=%u, b2=%u.\n",a[i]->DMnum,x,a1,a2,b1,b2);
*/

						if(((double)x / (double)(a2 - a1)) >= level1){
							dmID |= a[i]->dmlist[m].ID;
						}
					}
				}
			}
			a[i]->matchlist7[j].dmID = dmID;

			for(m=0;m<t1;m++) clearSQlist(&t[m]);
			t2 = 0;	
			for(k=0;k<n3;k++){
				if(ID == a[i]->sqlist3[k].ID && \
				a[i]->sqlist3[k].qID == a[i]->n0){
					copySQlist(&t[t2++],&a[i]->sqlist3[k]);
				}
			}
			m=FindNode(a,nodes,ID);

		/* June 17, 2007; corrected Aug. 23, 2007 */
			corr_length1 = a[i]->len;
			corr_length2 = a[m]->len;	
			/* conditions include nucleomorph */

			/* a[i] is eukaryotic, a[m] is prokaryotic */
			if((a[i]->kingdom == 3 || a[i]->kingdom == 19) && (a[m]->kingdom != 3 && a[m]->kingdom != 19)){
				for(k=0;k<a[i]->DMnum;k++){
					a1=a[i]->dmlist[k].start;
					a2=a[i]->dmlist[k].end;
					x = overlap(a1,a2,1,50);
					if(x > 0.1) break;
				}
				if(!(a[i]->matchlist7[j].dmID & a[i]->dmlist[k].ID)){
					corr_length1 -= (a[i]->dmlist[k].end - a[i]->dmlist[k].start + 1);	
					if(a[i]->corr_length > corr_length1) a[i]->corr_length = corr_length1;
				}
			}
			/* a[m] is eukaryotic, a[i] is prokaryotic */
			if((a[i]->kingdom != 3 && a[i]->kingdom != 19) && (a[m]->kingdom == 3 || a[m]->kingdom == 19)){
				start1 = 1;
				for(k=0;k<n3;k++){
					if(ID == a[i]->sqlist3[k].ID && a[i]->sqlist3[k].qID == a[i]->n0){
						if(a[i]->sqlist3[k].Qstart < _min_domain_size){
							if(a[i]->sqlist3[k].Sstart - a[i]->sqlist3[k].Qstart > (a[m]->len / 10)){
								start1 = a[i]->sqlist3[k].Sstart;
								break;
							}
						}
					}
				}	
				corr_length2 -= (start1 - 1);
				if(a[m]->corr_length > corr_length2) a[m]->corr_length = corr_length2;
			}

			a[i]->corr_length = corr_length1;
			
			overlap_value = OverlapScore2(t,t2,corr_length1,corr_length2);
			if(overlap_value > 1.0) overlap_value = OverlapScore2(t,t2,a[i]->len,a[m]->len);

			a[i]->matchlist7[j].overlap = overlap_value;

/*
printf("i=%u, j=%u, ID=%u, score=%10.2e, overlap=%7.3f, dmID=%lu.\n",i,j,a[i]->matchlist7[j].ID,a[i]->matchlist7[j].score,\
a[i]->matchlist7[j].overlap, a[i]->matchlist7[j].dmID);
*/

		}

		/* June 17, 2007 
		if(a[i]->kingdom == 3){
			counter0 = counter1 = counter2 = 0;
			for(j=0;j<n4;j++){
				ID = a[i]->matchlist7[j].ID;
				m=FindNode(a,nodes,ID);
				if(a[m]->kingdom != 3){
					counter0 += 1;
					if(!(a[i]->matchlist7[j].dmID & 0x1)) counter1 += 1;
					if(!(a[i]->matchlist7[j].dmID & 0x2)) counter2 += 1;
				}
				if(((double)counter1 / (double)counter0) > proklevel){
					delete1 = TRUE;
					if(((double)counter2 / (double)counter0) > proklevel){ 
						delete2 = TRUE;
					}
				}
	
				
			}
		}
		 not used. */


	}
	return a;
}
/******************************************************************/


/******************************************************************/
void CleanupIDall(Node **a,unsigned nodes)

/* In this subroutine, 0 is removed from idlist4 in all nodes*/
/******************************************************************/
{
	unsigned i;

	if(nodes == 0) return;
	for(i=1;i<=nodes;i++){
		CleanupID(a,nodes,i);
	}

	return;
}
/******************************************************************/


/******************************************************************/
void CleanupID(Node **a,unsigned nodes,unsigned i)

/* In this subroutine, 0 is removed from idlist4 */
/******************************************************************/
{
	unsigned j,k;
	unsigned n4;

	if(nodes == 0) return;
	if(i > nodes || i == 0) return;

	if((n4 = a[i]->n4) == 0) return;
	for(j=0;j<n4;j++){
		if((k=a[i]->idlist4[j])==0 || k > nodes){
			if(j < n4-1){
				a[i]->idlist4[j] = a[i]->idlist4[n4-1];
			}
			a[i]->idlist4[n4-1] = 0;
			n4--;
			a[i]->n4 = n4;
		}
	}

	return;
}
/******************************************************************/


/******************************************************************/
void RemoveID(Node **a,unsigned nodes,unsigned i,unsigned ID)

/* In this subroutine, ID is removed from idlist4 */
/******************************************************************/
{
	unsigned j,k;
	unsigned n4;

	if(nodes == 0) return;
	if(i > nodes || i == 0) return;
	if(ID == 0 || ID > nodes) return;

	n4 = a[i]->n4;
	for(j=0;j<n4;j++){
		if((k=a[i]->idlist4[j])==0 || k > nodes) continue;
		if(k==ID){
			if(j < n4-1){
				a[i]->idlist4[j] = a[i]->idlist4[n4-1];
			}
			a[i]->idlist4[n4-1] = 0;
			n4--;
			a[i]->n4 = n4;
			break;
		}
	}

	return;
}
/******************************************************************/


/******************************************************************/
void MakeClist(Node **a, unsigned nodes)

/* This subroutine makes ch list. */
/******************************************************************/
{
	unsigned i,k0,n4,k,j;


	/* Assigning parent nodes */
	for(i=1;i<=nodes;i++){ 
		if(!a[i]->active){
			k0 = a[i]->n0;		/* k0 is the N0 of node i */
			ch[k0] = k0;
			continue;
		}
		if(a[i]->n3 == 0 || (n4=a[i]->n4)==0) continue; 
		for(j=0;j<n4;j++){
			if((k = a[i]->idlist4[j]) == 0 || k > nodes) continue;		/* k is ID of members of idlist4 */
			if(ch[k] == 0) {
				ch[k] = a[i]->n0;
				continue;
			}
		}
	}
	return;
}
/******************************************************************/



/******************************************************************/
Node **MergeCliques(Node **a,unsigned nodes)

/* version 350
Duplicated cliques are merged. Criteria are:
1. If many of the members of a node i are members of another node ix,
and if node i is not a member of node ix, node i is added to idlist4 of
node ix. Then, a[i]->domain is set to 1, 2 or 4 depending on situation. */
/******************************************************************/
{
	unsigned i,j,k0,k,m,n4,n4x,n7,n7x;
	unsigned total,count0,count1;
	unsigned ix,kk,jj,kx;
	unsigned ixx;
	unsigned j4,k4;
	unsigned jx,jy;
	unsigned j1,j2;
	double min_overlap = 0.5;	/* This is different from int _min_overlap. */
	double overlap1, overlap2;
	double level1 = _level1 * 0.85;
	char flag = 0;
	Boolean nexti_flag = FALSE;
	Boolean to_add = FALSE;
	Boolean flag1 = FALSE;
	Boolean to_merge = FALSE;	/* 355 */

	printf("\nFinding duplicate nodes, and then trying to merge initial cliques ... \n");

	/* Finding duplication, and trying to merge cliques */
	for(i=1;i<=nodes;i++){ 
		PrintProgress(i,10,1000);

		if(!a[i]->active) continue;
		if(a[i]->n3 == 0 || (n4=a[i]->n4)<=1) continue; 
		k0 = a[i]->n0;
		n7 = a[i]->n7;

		ClearClist(nodes);
		MakeClist(a,nodes);

		if(ch[k0] != 0 && ch[k0] != k0){

			/* Normally, this should not occur. */
			/* check all members and finally set n4=0 for this node */
			printf("checkpoint 1 in MergeCliques. i=%u, %s\n",i,a[i]->name);
			continue;
		}

		flag = 0;
		nexti_flag = FALSE;
		for(j=0;j<n4;j++){
			if(nexti_flag) break;
			if((k = a[i]->idlist4[j]) == 0 || k > nodes) continue;

			/* ch[k] holds a tentative parent of node k */
			if((kx = ch[k]) == 0) continue;
			ix = FindNode(a,nodes,kx);
			if(ix == i || ix == 0) continue;
				/* In most normal cases, ix = i. */

				/* If ix != i, the list in ix is compared with the list in i. */
			n4x = a[ix]->n4;
			if(n4x == 0 || n4x > nodes){
				continue;
			}

/* 355L 
			to_merge = FALSE;
			if(use_org){
				switch(CountOrgInIDlist(a,nodes,i,ix)){
					case 0:
					case -1:
						to_merge = FALSE;
						break;
					case 1: 
					case 2: 
						to_merge = 1;
						break;
					case 3: 
						to_merge = 2;
						break;
					default:
						to_merge = FALSE;
						break;
				}
			}
This is not a good code. */

			flag = 4;	/* default: group i is to be decomposed. */

			count1 = 0;
			count0 = 0;
			total = 0;
			for(jj=0;jj<n4;jj++){
				if((kk = a[i]->idlist4[jj]) == 0 || kk > nodes) continue;
				total += 1;
				to_add = FALSE;
				for(jx=0;jx<n4x;jx++){
					if(kk == a[ix]->idlist4[jx]){
						to_add = TRUE;
						break;
					}
				}
				if(jx == n4x){
					if(ch[kk] == kx) to_add = TRUE;	/* may be redundant */
					else if(ch[kk] == k0 && ch[k0] == kx) to_add = TRUE;
					else if(ch[kk] == kk){
						ixx = FindNode(a,nodes,kk);
						if(a[ixx]->n4 == 1) to_add = TRUE;
					} 
				}
				if(to_add) count1 += 1;
				else count0 += 1;
			}
			if(count1 + count0 != total){
				printf("checkpoint 2 in MergeCliques.\n");
				count0 = total - count1;
			}

			if(count1 >= total * level1 || count1 + 1 >= total) flag = 1;
			else {	
				for(j1=0;j1<num_thr;j1++) {
					if(thr_list[j1] >= a[i]->final_thr) break;
				}
				for(j2=0;j2<num_thr;j2++) {
					if(thr_list[j2] >= a[ix]->final_thr) break;
				}
			/* clique i is conserved, while components of i are removed from ix. */
				if(j2 > j1 + 1) flag = 2;
				else {
					if(j2 <= j1 + 1 && j2 + 1 >= j1) flag = 5;
					overlap1 = 1.0;
					n4 = a[i]->n4;
					n7 = a[i]->n7;
					for(j1=0;j1<n4;j1++) {
						for(j2=0;j2<n7;j2++) {
							if(a[i]->idlist4[j1] == a[i]->matchlist7[j2].ID) {
								if(overlap1 > a[i]->matchlist7[j2].overlap) overlap1 = a[i]->matchlist7[j2].overlap;
							}
						}
					}
					n4x = a[ix]->n4;
					n7x = a[ix]->n7;
					overlap2 = 1.0;
					for(j1=0;j1<n4x;j1++) {
						for(j2=0;j2<n7x;j2++) {
							if(a[ix]->idlist4[j1] == a[ix]->matchlist7[j2].ID) {
								if(overlap2 > a[ix]->matchlist7[j2].overlap) overlap2 = a[ix]->matchlist7[j2].overlap;
							}
						}
					}
					if(overlap1 > overlap2 + 0.1) flag = 2;
					else if(overlap1 + 0.2 < overlap2) flag = 1;
					else if((abs(overlap1 - overlap2) < 0.2) && flag==5) flag = 1;
					else flag = 4;
				}
			}
	
			/* 355L 
			if(to_merge == 2){
				flag = 1;	
				printf("Merging cluster %s n4=%u to cluster %s n4x=%u.\n",a[i]->name,a[i]->n4,a[ix]->name,a[ix]->n4);
			}	
			This is not a good choice.*/

			switch (flag) {	
				case 1:
					if((a[ix]->idlist4=(unsigned*)realloc(a[ix]->idlist4,(n4x+count0+1)*sizeof(unsigned)))==NULL){
						fprintf(stderr,"Memory allocation error in MergeCliques 1.\n");
						exit(1);
					}
					for(jj=n4x;jj<n4x+count0;jj++) a[ix]->idlist4[jj] = 0;


/* debug 

printf("i=%u, n4=%u\n   idlist=",i,n4);
for(jj=0;jj<n4;jj++) printf(" %u",a[i]->idlist4[jj]);
printf("\n");
printf("ix=%u, n4x=%u\n   idlist=",ix,n4x);
for(jj=0;jj<n4x;jj++) printf(" %u",a[ix]->idlist4[jj]);
printf("\n");
*/


					for(jj=0;jj<n4;jj++){
						kk = a[i]->idlist4[jj];
/*
printf("kk=%u. \n",kk);
*/
						if((kk = a[i]->idlist4[jj]) == 0 || kk > nodes) continue;
/*
if(kk==534 || kk==1108){
printf("i=%u, n4=%u\n   idlist=",i,n4);
for(temp=0;temp<n4;temp++) printf(" %u",a[i]->idlist4[temp]);
printf("\n");
}
*/
						if(ch[kk] == kx){
							RemoveID(a,nodes,i,kk);
							n4--;
							jj--;
							continue;
						} else if(ch[kk] == k0){
							ch[kk] = kx;
							a[ix]->idlist4[n4x++] = kk;
							RemoveID(a,nodes,i,kk);
							n4--;
							jj--;
	
					/* check overlap in matchlist7 to see if it is a fragment */
							m=FindNode(a,nodes,kk);
							for(jy=0;jy<n7;jy++){
								if(a[i]->matchlist7[jy].ID == kk){
									if(a[i]->matchlist7[jy].overlap < min_overlap && a[m]->len < a[ix]->len ){
										 a[m]->domain |= 0x02;
									}
									break;
								}
							}
							continue;
						} else {
	
						/* may be multidomain */

							RemoveID(a,nodes,i,kk);
							n4--;
							jj--;
						}
					}
					a[i]->n4 = 0;
					a[i]->n3b = a[i]->n3;
					a[i]->n3 = 0;
					a[ix]->n4 = n4x;
					nexti_flag = TRUE;
					flag = 0;
					break;		/* next i*/

			/* clique i is conserved, while components of i are removed from ix. */
				case 2: 
					for(jj=0;jj<n4;jj++) {
						kk = a[i]->idlist4[jj];
						for(jx=0;jx<n4x;jx++){
							if(a[ix]->idlist4[jx] == kk) {
								RemoveID(a,nodes,ix,kk);
								n4x--;
								break;
							}
						}
					}
					a[ix]->n4 = n4x;
					break;

				case 3:
					break;

				case 4:
					CleanupID(a,nodes,i);
					/* separate all members of a[i] */
					n4=a[i]->n4;
					printf("Node %u %s is being decomposed:",i,a[i]->name);	
					
/* for debug */
printf("i=%u, n4=%u\n   idlist=",i,n4);
for(jj=0;jj<n4;jj++) printf(" %u",a[i]->idlist4[jj]);
printf("\n");
printf("ix=%u, n4x=%u, kx=%u\n   idlist=",ix,n4x,kx);
for(jj=0;jj<n4x;jj++) printf(" %u",a[ix]->idlist4[jj]);
printf("\n");
					
					
					for(j4=0;j4<n4;j4++){
						if((k4 = a[i]->idlist4[j4]) == 0 || k4 > nodes) continue;
						if(ch[k4] == kx){
							a[i]->idlist4[j4] = 0;
							continue;
						}
						ix = FindNode(a,nodes,k4);
						printf(" -%u-",a[ix]->n0);

						/* looking for item k4 in other clusters */
						flag1 = FALSE;

						for(ixx=1;ixx<=nodes;ixx++){
							if(a[ixx]->n3 == 0 || a[ixx]->n4 == 0) continue;
							if(ixx == i || ixx == ix || a[ixx]->n0 == kx) continue;
							for(jj=0;jj<a[ixx]->n4;jj++){
								if(k4 == a[ixx]->idlist4[jj]){
									flag1 = TRUE;
									break;
								}
							}
							if(flag1) break;
						}


/*
						for(jj=0;jj<a[ix]->n3b;jj++){
							kk = a[ix]->sqlist3[jj].ID;
							if(kk == 0 || kk > nodes || kk == k4) continue;
							ixx = FindNode(a,nodes,kk);
							if(ixx == 0 || ixx > nodes) continue;
							if(a[ixx]->n4 != 0){
								for(j5=0;j5<a[ixx]->n4;j5++){
									if(k4 == a[ixx]->idlist4[j5]){
										flag1 = TRUE;
										break;
									}
								}
							} else {
								kp = a[ixx]->pID;
								ixxp = FindNode(a,nodes,kp);
								if(ixxp == 0 || ixxp > nodes) continue;
								if(a[ixxp]->n4 != 0){
									for(j5=0;j5<a[ixxp]->n4;j5++){
										if(k4 == a[ixxp]->idlist4[j5]){
											flag1 = TRUE;
											break;
										}
									}
								}
							}

							if(flag1) break;
						}
*/


						if(flag1){
							a[i]->idlist4[j4] = 0;
							continue;
						}
						
						if(a[ix]->n4 == 0){
							if(a[ix]->idlist4[0] != a[ix]->n0){
								for(jj=1;jj<a[ix]->n4+10;jj++){
									if((kk=a[ix]->idlist4[jj]) == 0 || kk > nodes) break;
									if(kk == a[ix]->n0){
										a[ix]->idlist4[jj] = a[ix]->idlist4[0];
										a[ix]->idlist4[0] = a[ix]->n0;
									}
								}
							}
						}
						a[ix]->idlist4[0] = a[ix]->n0;
						a[ix]->n4 = 1;
						a[ix]->n3 = a[ix]->n3b;
/*
 * printf(" ix: %s n3=%u n3b=%u n30=%u\n",a[ix]->name,a[ix]->n3, a[ix]->n3b,a[ix]->n30);
*/
					}
					printf("\n");
					a[i]->n4b = a[i]->n4;
					a[i]->n4 = 1;
					a[i]->idlist4[0] = a[i]->n0;

/*
 * printf(" i: %s n3=%u n3b=%u n30=%u\n",a[i]->name,a[i]->n3, a[i]->n3b,a[i]->n30);
*/					
					nexti_flag = TRUE;
					break;

				default:
					break;
			}
		}
	}

	return a;
}
/******************************************************************/


/******************************************************************/
Node **CreateCliques(Node **a,unsigned nodes)

/* This is new from version 3.5. Modified 3.5.1.
For each i, items are accumulated recurrently by incorporating 
items cited in sqlist3 as in MergeList. However, not all SQlists
are merged. Here, thr is scanned from thr_initial to see if
members form a clique. If yes, thr is increased. Otherwise, 
thr is decreased (more stringent), and clustering is tried again.
There are different conditions for incorporating members.
	1. E-value
	2. overlap score
	3. domain composition.
Clique formation is judged according to p-quasi complete
condition, namely, 
	1. a proportion (p) of total member have
link to all other members.
	2. each member has at least two links to other members.
In this subroutine, idlist4 is also made along with sqlist3.
*/
/******************************************************************/
{
	unsigned i,j,j4,j7,k,m,n,j4x;
	unsigned c;
	unsigned len;
	unsigned n3,n4,n7,n4delta;
	unsigned N2;
	double thr_bak;
	double level1 = _level1;
	double level2 = _CC_level_2;
	double level3 = _CC_level_3;
	double min_thr = _CC_min_thr;
	unsigned min_count = 2;
	unsigned long maskID;
	unsigned total,count;
	unsigned min_domain_size = _min_domain_size * 1.2;
	int mat_limit;
	int x1,y2;

/*
	double level2 = 0.9;
	double level3 = 0.2;
	double min_thr = 1e-40;
*/

	printf("CreateCliques stage 1: creating initial idlist4.\n");
	fflush(stdout);

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(!a[i]->active) continue;		/* 308 */
		if((n3=a[i]->n3) == 0) continue;

		/* This part is not done. Inactive nodes are excluded above. */
		if(a[i]->domain != 0){
			if(print_2D_tables) printf("Skipping node i=%u, a multidomain or large protein.\n",i);
			continue;
		}

		ClearClist(nodes);
		ch[a[i]->n0] = TRUE;

/* 351 */
		mat_limit = print_2D_matrix(a,nodes,i);
		
		x1 = mat_limit % 11;
		y2 = mat_limit / 11;
		if(print_2D_tables) printf("Best local maximum: x1 = %d, y2 = %d, overlap = %3.1f, thr = %6.1e.\n",x1,y2,0.1*x1,thr_list[y2]);
		thr_bak = thr;
		thr = thr_list[y2];
		if(print_2D_tables){
			printf("thr = %6.1e \n",thr);
			fflush(stdout);
		}
		
/* 350 
		thr_selected = best_thr(a[i]->sqlist3,n3);

		thr_bak = thr;
		thr = thr_list[thr_selected];
		printf("thr = %6.1e \n",thr);
*/
		a[i]->final_thr = thr;
		a[i]->overlap = 0.1 * x1;
		
		if(clique_mode){
			MakeIDlist_clique(a,nodes,i);
		} else MakeIDlist_i(a,nodes,i);

		thr = thr_bak;
	}

 /* In org mode, optimal selection of homologs must be achieved in stage 1
  * by the 2D table method. 
	if(use_org){
		printf("\nCreateCliques stage 2 is omitted in org mode.\n");
		goto STAGE3;
	}

*/

	printf("\nCreateCliques stage 2: optimizing idlist4.\n");
	fflush(stdout);

	for(i=1;i<=nodes;i++){

		PrintProgress(i,10,1000);
		if(!a[i]->active) continue;		/* 308 */
		if((n3=a[i]->n3) == 0) continue;
		if(a[i]->domain != 0) continue;
		if(a[i]->final_thr < min_thr || a[i]->overlap >= 0.88 * level1) continue;	/* 351 */

		n4 = a[i]->n4;
		n7 = a[i]->n7;
		len = a[i]->len3;	/* size of sqlist3 */

		for(j=0;j<n4;j++){
			j4 = a[i]->idlist4[j];

			if(j4 == a[i]->n0 || j4 == 0) continue;

		/* check for levels of thr */
			
			j4x = FindNode(a,nodes,j4);
			if(j4x == 0 || j4x > nodes) {
				fprintf(stderr,"item %u is strange. Skipping ...\n",j4);
			} else if(!use_org) {	/* 352 */
				/* 351 */
				if(thr_level(a[j4x]->final_thr) + 3 < thr_level(a[i]->final_thr) || \
								a[j4x]->DMnum != a[i]->DMnum){ 
					a[i]->idlist4[j]=a[i]->idlist4[n4-1];
					a[i]->idlist4[n4-1] = j4;
					n4--;
					a[i]->n4 = n4;
					j--;
					continue;
				}
			}

			
		/* check for overlap and domain */

			for(m=0;m<n7;m++){
				if(j4 == a[i]->matchlist7[m].ID) break;
			}
			if(m == n7){
				fprintf(stderr,"item was not found in matchlist7 in CreateCliques1.\n");
				fprintf(stderr,"i=%u, ID=%u, n4=%u, n7=%u.\n",i,j4,n4,n7);
				continue;
			}
			j7 = m;

	/*		if(a[i]->matchlist7[j7].overlap < level1){
	*/
			if(a[i]->matchlist7[j7].overlap < a[i]->overlap){
				a[i]->idlist4[j]=a[i]->idlist4[n4-1];
				a[i]->idlist4[n4-1] = j4;
				n4--;
				a[i]->n4 = n4;
				j--;
				continue;
			}
			
			maskID = a[i]->maskID;

			/* If maskID is not given in data.out */
			if(maskID == 0x00){
				for(m=0;m<a[i]->DMnum;m++){
					if((a[i]->dmlist[m].end - a[i]->dmlist[m].start) < min_domain_size) continue;
					maskID |= a[i]->dmlist[m].ID;
				}
				a[i]->maskID = maskID;
			}

			if((total = CountBits(maskID)) < 2) continue;
			count = CountBits(a[i]->matchlist7[j7].dmID & maskID);
			if((double)count < (double)total * level2){
/*				fprintf(stderr,"item is removed from idlist4 in CreateCliques2.\n");
*/
				a[i]->idlist4[j]=a[i]->idlist4[n4-1];
				a[i]->idlist4[n4-1] = j4;
				n4--;
				a[i]->n4 = n4;
				j--;
				continue;
			}
		}
	}

	/* At this stage, the idlist4 of each node has been optimized. */
	/* In the next section, the idlist4 is merged. */

STAGE3:

	printf("\nCreateCliques stage 3: merging idlist4: Removed items are listed below.\n");
	fflush(stdout);

	for(i=1;i<=nodes;i++){

		PrintProgress(i,10,1000);
		if(i % 1000 == 0) printf("\n");

		if(!a[i]->active) continue;	
		if((n3=a[i]->n3) == 0) continue;
		if(a[i]->domain != 0) continue;
		maskID = a[i]->maskID;
		/* If maskID is not given in data.out */
		if(maskID == 0x00){
			for(m=0;m<a[i]->DMnum;m++){
/*				if((a[i]->dmlist[m].end - a[i]->dmlist[m].start) < min_domain_size * 2) continue;
*/
				if((a[i]->dmlist[m].end - a[i]->dmlist[m].start) < min_domain_size) continue;
				maskID |= a[i]->dmlist[m].ID;
			}
			a[i]->maskID = maskID;
		}

		n4 = a[i]->n4;
		n7 = a[i]->n7;

		for(j=0;j<n4;j++){
			j4 = a[i]->idlist4[j];
			if(j4 == a[i]->n0 || j4 == 0) continue;
			c = FindNode(a,nodes,j4);
			if(c == i || c > nodes || c == 0) continue;	/* skip self */
			if(a[c]->n3 == 0 || a[c]->n4 == 0) continue;

		/* checking if each of the members of a[c] is a member of a[i] */
		/* just checking idlist4 */

			if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n4 + a[c]->n4 + 1)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in CreateCliques3.\n");
				exit(1);
			}
			for(k=n4;k<n4+a[c]->n4;k++) a[i]->idlist4[k] = 0;
			n4delta = 0;
			for(k=0;k<a[c]->n4;k++){
				N2=a[c]->idlist4[k];
				n=FindNode(a,nodes,N2);
				if(a[n]->domain != 0) continue;
				for(m=0;m<n4;m++){
					if(a[i]->idlist4[m] == N2) break;
				}
				if(m < n4) continue;
				a[i]->idlist4[n4 + n4delta++] = N2;
				if(n4delta > a[c]->n4){
					fprintf(stderr,"Error in n4delta in CreateCliques4.\n");
					break;
				}
			}
			if(n4delta > 0 && (double)n4delta <= (double)(a[c]->n4) * level3 + (double)min_count){
	
				for(k=n4;k<=n4+n4delta;k++){
					N2=a[i]->idlist4[k];
					n=FindNode(a,nodes,N2);
					if(n==0 || a[n]->domain != 0) continue;
					for(m=0;m<n7;m++){
						if(N2 == a[i]->matchlist7[m].ID) break;
					}
					if(m == n7) continue;
					j7 = m;

					if(a[i]->matchlist7[j7].overlap > level1) continue;
				
					if((total = CountBits(maskID)) < 2) continue;
	
					count = CountBits(a[i]->matchlist7[j7].dmID & maskID);
					if((double)count < (double)total * level2){
/*						printf("Removing an item: N0=%u, target=%u, n4=%u, n4delta=%u.\n",a[i]->n0,N2,n4,n4delta);
*/
						printf(" -%s ",a[i]->name);
						a[i]->idlist4[k]=a[i]->idlist4[n4+n4delta-1];
						if(n4delta == 0) break;
					    else n4delta--;
					}
				}

				
				n4 += n4delta;
				if(n4 > nodes) n4 = a[i]->n4;
				a[c]->final_thr = thr;
			}

			if(a[c]->n3 != 0) a[c]->n3b = a[c]->n3;
			a[c]->n3 = 0;
			a[c]->n4 = 0;
		}
		if(n4 > nodes) fprintf(stderr,"Strange value of n4=%u.\n",n4);
		a[i]->n4 = n4;
		if(a[i]->n3 != 0) a[i]->n3b = a[i]->n3;

	}
	printf("\n");

	return a;
}
/**********************************************/


/**********************************************/
Node **MergeList(Node **a,unsigned nodes)
/* Here, sqlist3 is modified by incorporating all items
cited in a[j]->sqlist3 referenced from a[i]->sqlist3.
In contrast to previous versions, n3 is not set to 0
in this subroutine. */
/* Now, multidomain is excluded to enable repeating
clustering with thr unchanged. */
/**********************************************/
{
	unsigned i,j,k,m,n,last;
	unsigned c;
	unsigned len;
	unsigned n3;
	unsigned N2;
	
	a[0]->n3 = 0;	/* flag */

	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(!a[i]->active) continue;		/* 308 */
		if((n3=a[i]->n3) == 0) continue;
		if(a[i]->final_thr < thr) continue;
		if(a[i]->domain != 0) continue;

		ClearClist(nodes);
		ch[a[i]->n0] = TRUE;

		len = a[i]->len3;	/* size of sqlist3 */
		for(j=0;j<n3;j++){
			c = FindNode(a,nodes,a[i]->sqlist3[j].ID);
			if(c==i || c>nodes || c==0) continue;	/* skip self */
			if(a[c]->n3 == 0) continue;
			if(a[c]->final_thr < thr || (a[c]->domain != 0)){
				last = n3 - 1;
				if(j<last){
					copySQlist(&a[i]->sqlist3[j],&a[i]->sqlist3[last]);
				}
				clearSQlist(&a[i]->sqlist3[last]);
				n3 -= 1;
				a[i]->n3 = n3;
				if(last == j) break;
				j--;
				continue;
			}
	
			ch[a[i]->sqlist3[j].ID] = TRUE;	
			for(k=0;k<a[c]->n3;k++){
				N2=a[c]->sqlist3[k].ID;
				if(ch[N2]){
					for(m=0;m<n3;m++){
						if(a[c]->sqlist3[k].ID == a[i]->sqlist3[m].ID && \
							a[c]->sqlist3[k].qID == a[i]->sqlist3[m].qID) break;
					}
					if(m<n3) continue;
				} else {
					m = n3;
				}
					
					ch[N2] = TRUE; 
					n=FindNode(a,nodes,a[c]->sqlist3[k].ID);
					if(a[n]->final_thr < thr || a[n]->domain != 0) continue;
					if(m > len - 1){
						len = len + a[i]->n1 + 1;
						a[i]->len3 = len;
						if((a[i]->sqlist3=(SQlist*)realloc(a[i]->sqlist3,len*sizeof(SQlist)))==NULL){
							fprintf(stderr, "Memory allocation error in MergeList\n.");
							fflush(stderr);
							return NULL;
						}
						for(n=n3;n<len;n++) clearSQlist(&a[i]->sqlist3[n]);
					}
					copySQlist(&a[i]->sqlist3[n3],&a[c]->sqlist3[k]);
					a[i]->n3 = ++n3;
					a[0]->n3 += 1;	
			/* this indicates number of items that have been changed during the call of this function. */

			}
			if(a[c]->n3 != 0) a[c]->n3b = a[c]->n3;
			a[c]->n3 = 0;
		}
		if(a[i]->n3 != 0) a[i]->n3b = a[i]->n3;
	}
	return a;
}
/**********************************************/


/***************************************************/
void MakeSublist(Node **a,unsigned nodes,unsigned i)
/* Make sublist for the parent i */
/***************************************************/
{
	unsigned n3,n4,n6,t1,t2,tmp;
	unsigned j1,j2,j3,k1,k2,k3;
	unsigned max,region,region1;
	unsigned maxn5,maxj1;
	unsigned ct;
	double score;
	SQlist *s,*t;
	char addtolist;		/* flag for adding to list. 1 or 0. */

	if(i>nodes) return;
	if(a[i]->n3==0) return;
	n4 = a[i]->n4;
	n3 = a[i]->n3b;
	if(n4 < 3){
		a[i]->n5 = 0;
		a[i]->n6 = 0;
		return;
	}

	for(j1=0;j1<n4;j1++){
		k1=FindNode(a,nodes,a[i]->idlist4[j1]);
		if(k1==0) continue;
		if(n3 < a[k1]->n3b) n3 = a[k1]->n3b;
	}

	ClearRegion(R,n3);
	ClearRegion(R1,n3);
	ClearRegion(R2,n3);

	if((s=(SQlist*)calloc(n3,sizeof(SQlist)))==NULL){
		fprintf(stderr,"Error in MakeSublist1.\n");
		exit(1);
	}
	if((t=(SQlist*)calloc(n3,sizeof(SQlist)))==NULL){
		fprintf(stderr,"Error in MakeSublist1-1.\n");
		exit(1);
	}
	if((a[i]->reflist5=(unsigned*)realloc(a[i]->reflist5,(n4+1)*sizeof(unsigned)))==NULL){
		fprintf(stderr,"Memory allocation error in MakeSublist2.\n");
		exit(1);
	}
	for(j1=0;j1<n4;j1++){
		a[i]->reflist5[j1] = a[i]->idlist4[j1];
	}
	a[i]->n5 = n4;
	a[i]->n6 = 0;

	for(j1=0;j1<n4;j1++){
		k1=FindNode(a,nodes,a[i]->reflist5[j1]);

/*		if(a[k1]->n3b < 3) continue;
*/

		if(a[k1]->n3b < 2) continue;


		if(k1==0) continue;
		if(a[k1]->n6 == 0){ 
			n6 = a[k1]->n3b;
			if((a[k1]->sublist6=(unsigned*)realloc(a[k1]->sublist6,(n6+1)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in MakeSublist3.\n");
				exit(1);
			}
			ct=1;
			a[k1]->sublist6[0]=a[k1]->n0;
		} else {
			n6 = a[k1]->n3b;
			if(n6 > a[k1]->n6){
				if(a[k1]->sublist6==NULL){
					if((a[k1]->sublist6=(unsigned*)calloc(n6+1,sizeof(unsigned)))==NULL){
						fprintf(stderr,"Memory allocation error in MakeSublist3-1.\n");
						exit(1);
					}
				} else {
					if((a[k1]->sublist6=(unsigned*)realloc(a[k1]->sublist6,(n6+1)*sizeof(unsigned)))==NULL){
						fprintf(stderr,"Memory allocation error in MakeSublist3-2.\n");
						exit(1);
					}
				}
			}
			ct=a[k1]->n6;
		}



/* for debug */
		printf("j1=%u, k1=%u, %s\n",j1,k1,a[k1]->name);

	/* getting the largest region of homology */
		region = 0;
		max = 0;
		for(j2=0;j2<n3;j2++){
			if(a[k1]->n0 != a[i]->sqlist3[j2].qID) continue;
			k2=FindNode(a,nodes,a[i]->sqlist3[j2].ID);
			if(k1==k2 || k2==0) continue;
					/* reconsider items with n6 == 1. */
			if(a[k2]->domain != 0 || a[k2]->n6 > 1) continue; /* skip multidomain protein */
			for(j3=0;j3<n4;j3++){	/* skip items already recognized as a member of a subgroup. */
				if(a[i]->reflist5[j3] == a[k2]->n0){
					break;
				}
			}
			if(j3==n4) continue;

			region1 = GetRegion(a[i]->sqlist3,j2,n3);
			if(region1 > region){
				region=region1;
				max = j2;
			}
		}

/* for debug */
printf("i=%u, j1=%u, max=%u, region=%u, n3=%u, k2=%u.\n",i,j1,max,region,n3,k2);
fflush(stdout);

		if(region <= _minlevel * a[k1]->len){
			a[k1]->n6=1;	/* potential group head */
			if(a[k1]->sublist6[0]==0) a[k1]->sublist6[0]=a[k1]->n0;
			printf(".....skip\n");
			continue; 	/* next j1 */
		}

		t2=0;
		for(j3=0;j3<n3;j3++){
			if(a[i]->sqlist3[j3].ID == a[i]->sqlist3[max].ID && \
			a[i]->sqlist3[j3].qID == a[k1]->n0){
				copySQlist(&t[t2++],&a[i]->sqlist3[j3]);
			}
		}
		k3=FindNode(a,nodes,a[i]->sqlist3[max].ID);
		score = OverlapScore2(t,t2,a[k1]->len,a[k3]->len);
		if(score > _matchlevel){
			a[k1]->sublist6[ct++] = a[i]->sqlist3[max].ID;
			a[k3]->n6=0;		/* now a member not a group head */
			for(j3=0;j3<n4;j3++){
				if(a[i]->reflist5[j3] == a[i]->sqlist3[max].ID){
					a[i]->reflist5[j3]=0;
					break;
				}
			}
		}
		else{
			if(a[k3]->n6==0){
				a[k3]->n6=1;	/* potential group head */
			}
			if(a[k3]->sublist6[0]==0) a[k3]->sublist6[0]=a[k3]->n0;
			continue; 	/* next j1 */
		}

	/* adding other members */

		for(j2=0;j2<n3;j2++){
			if(a[k1]->n0 != a[i]->sqlist3[j2].qID) continue;
			k2=FindNode(a,nodes,a[i]->sqlist3[j2].ID);
			if(k1==k2 || k2==0) continue;
					/* if n6 == 1, reconsidered. */
			if(a[k2]->domain != 0 || a[k2]->n6 > 1) continue; /* skip multidomain protein */
			for(j3=0;j3<n4;j3++){	/* skip items already recognized as a member of a subgroup. */
				if(a[i]->reflist5[j3] == a[i]->sqlist3[j2].ID){
					break;
				}
			}
			if(j3==n4) continue;
			for(j3=0;j3<ct;j3++){
				if(a[k1]->sublist6[j3] == a[i]->sqlist3[j2].ID){
					break;
				}
			}
			if(j3<ct) continue;

			addtolist = 0;
			t1=0;
			for(j3=0;j3<n3;j3++){
				if(a[i]->sqlist3[j3].ID == a[i]->sqlist3[j2].ID && \
				a[k1]->n0 == a[i]->sqlist3[j2].qID){
					copySQlist(&s[t1++],&a[i]->sqlist3[j3]);
				}
			}
			if(t1==0) continue;
			region = GetRegion2(s,t1,t,t2);
			score = (double)region / (double)a[k1]->len;
			if(score < _matchlevel){
				if(a[k2]->n6 == 0){
					a[k2]->n6 = 1;	/* potential group head */
				}
				if(a[k2]->sublist6[0]==0) a[k2]->sublist6[0]=a[k2]->n0;
				continue;
			}
			else addtolist = 1;

			if(addtolist){
				if(a[k2]->n6!=0){
					a[k2]->domain |= 0x1;
					continue;
				}
				a[k1]->sublist6[ct++]=a[i]->sqlist3[j2].ID;
	printf("   member %u added to a[%u]->sublist6.\n",ct,k1);
				for(j3=0;j3<n4;j3++){
					if(a[i]->reflist5[j3] == a[i]->sqlist3[j2].ID){
						a[i]->reflist5[j3]=0;
						break;
					}
				}
				if(j3==n4){
					fprintf(stderr,"Item %s already deleted in a[%u]->sublist6.\n",a[k2]->name,i);
				}
			}
		}

		a[k1]->n6=ct;
	}

/* The largest subgroup comes first. Not a real sorting. */

	maxn5 = 0;
	maxj1 = 0;
	for(j1=0;j1<n4;j1++){
		k1=FindNode(a,nodes,a[i]->reflist5[j1]);
		if(k1==0) continue;
		if(maxn5 < a[k1]->n6){
			maxn5 = a[k1]->n6;
			maxj1 = j1;
		}
	}
	if(maxn5 != 0 && maxj1 != 0){
		tmp = a[i]->reflist5[0];
		a[i]->reflist5[0] = a[i]->reflist5[maxj1];
		a[i]->reflist5[maxj1] = tmp;
	}
	return;
}
/***************************************************/



static unsigned current_n4;
static unsigned current_n4b;

/***************************************************/
int cmpIDlist(Node **a,unsigned nodes,int it)
/* returns TRUE if multidomain is found, and the
iteration is repeated with thr value unchanged. */
/***************************************************/
{
	unsigned i,j,k,m,n,kk;
	double increase=1.0;
	double thr1 = thr_list[it-1];
	unsigned last;
	unsigned real_n4b;
	int increment;
	SQlist *tmp;
	static unsigned *vector;
	static unsigned vect_dim=0;
	Boolean found_multidomain=FALSE;
	double _max_increase = 3.0;	/* maximum increase ratio to proceed to domain identificaiton */
	unsigned _max_increment = nodes / 10;	/* maximum increment to proceed to domain identificaiton */
	double slope = 0.0;

	printf("Comparing ID lists\n");
	fflush(stdout);

	if(it==0){
		printf("Making backup of IDlists.\n");
		for(i=1;i<=nodes;i++){
			if(!a[i]->active) continue;		/* 308 */
			backupIDlist(a,nodes,i);
			clearIDlist(a,nodes,i);
		}
		PrintN4(a,nodes,it);
		fflush(stdout);
		return FALSE;
	} else if(it==num_thr+1){
		printf("Finishing the last IDlists.\n");
		for(i=1;i<=nodes;i++){
			if(!a[i]->active) continue;		/* 308 */
			if(a[i]->domain != 0){
				a[i]->n3 = 1;
				for(j=1;j<a[i]->n3b;j++){
					clearSQlist(&a[i]->sqlist3b[j]);
				}
				a[i]->n3b=1;
				recoverIDlist(a,nodes,i);
				continue;
			}

			if(a[i]->final_thr < thr){
				a[i]->n3 = a[i]->n3_bak;
				a[i]->n3b = a[i]->n3b_bak;
				if(a[i]->n3==0) continue;
				tmp = a[i]->sqlist3;
				a[i]->sqlist3 = a[i]->sqlist3b;
				a[i]->sqlist3b = tmp;
			}
		}
		PrintN4(a,nodes,it);
		fflush(stdout);
		return FALSE;
	}

	if(vect_dim ==0){
		if((vector=(unsigned*)calloc(100,sizeof(unsigned)))==NULL){
			fprintf(stderr,"Error in memory allocation in cmpIDlist1.\n");
			exit(1);
		}
		vect_dim=100;
	}

/* for debug */
	PrintN4(a,nodes,it);

	for(i=1;i<=nodes;i++){
		PrintProgress(i,1,1000);
		if(!a[i]->active) continue;		/* 308 */
		if(a[i]->final_thr < thr || a[i]->domain != 0) continue;
		if(a[i]->n3 == 0) continue;
		if(a[i]->n4 < 6){ 	/* small groups are not compared, but the processing will be done later */
			continue;
		}

/* search for real n4b */
		real_n4b = 0;
		for(j=0;j<a[i]->n4;j++){
			k=FindNode(a,nodes,a[i]->idlist4[j]);
			if(real_n4b < a[k]->n4b){
				real_n4b = a[k]->n4b;
			}
		}
		increment = a[i]->n4 - real_n4b;
		increase = (double)a[i]->n4 / (double)real_n4b;

/* if new n4 is fairly larger than the previous n4 */
		slope = 0.7 * (double)number_of_genomes / _min_increase;
/*
		if(increase < _min_increase || increment < number_of_genomes * 0.7){
*/
	/* use of slope in 308b7 */
		if(increment + slope * increase > 1.4 * (double)number_of_genomes) {
			continue;
		}

/* here, a more rigourous test for homogeneity of groups is now implemented. */
	/* 	1. identification of multidomains.
		2. removal of multidomains from idlist4 and sqlist3
		3. similarity of length.
		4. similarity of overlapped region.
		
		In addition, multidomain is identified at this stage.
	*/

		if(increase < _max_increase && increment < _max_increment){

			SearchMultiDomain(a,nodes,i);

			if(RemoveMultiDomain(a,nodes,i)){

			/* RemoveMultiDomain == true */
			/* The a[i]->idlist4, in which multidomain was found, will not be processed. */

				found_multidomain = TRUE;
				continue;
	
			} else {

				if(vect_dim < a[i]->n4){
					if((vector=(unsigned*)realloc(vector,(a[i]->n4 + 2) * sizeof(unsigned)))==NULL){
						fprintf(stderr,"Error in memory allocation in cmpIDlist2.\n");
						exit(1);
					}
					vect_dim = a[i]->n4;
				}

				/* tTest returns true if new list4 should be aborted */	
				/* This has been erroneously reversed. Now corrected in 308b7 */
				if(!tTest_idlist4(a,nodes,i,vector,vect_dim)){
					continue;
				}
			}
		}

	/* RemoveMultiDomain == false and tTest == false */
	/* new idlist is judged to be aborted. Fixing the idlist at the previous stage.
		Individual items might be made fragment or multidomain. */

		a[i]->final_thr = thr1;
		for(j=0;j<a[i]->n4;j++){
			k=FindNode(a,nodes,a[i]->idlist4[j]);
			if(k==0) continue;
			if(k==i) continue;
			if(a[k]->final_thr < thr1) continue;
			a[k]->final_thr=thr1;
		}
	}

	/* Begin this round again after removing multidomains. */
	if(found_multidomain) return TRUE;


/* second cycle */
	for(i=1;i<=nodes;i++){
		if(!a[i]->active) continue;		/* 308 */
		if(a[i]->final_thr > thr1 * 1.001 || a[i]->final_thr < thr1 * 0.999 || a[i]->n4==0) continue;
		if(a[i]->n3b == 0) continue;	/* This is equivalent to final_thr=thr1 */
		for(j=0;j<a[i]->n4;j++){
			k=a[i]->idlist4[j];
			if(k==0) continue;
			kk=FindNode(a,nodes,k);
			if(kk==0) continue;
			if(a[kk]->final_thr < thr1) continue;
			a[kk]->final_thr = thr1;
		}
	}

/* third cycle */
	for(i=1;i<=nodes;i++){
		if(!a[i]->active) continue;		/* 308 */
		if(a[i]->final_thr > thr1 * 1.001 || a[i]->final_thr < thr1 * 0.999 || a[i]->n4==0) continue;
		if(a[i]->n3b == 0) continue;	/* This is equivalent to final_thr=thr1 */
		for(j=0;j<a[i]->n4;j++){
			k=a[i]->idlist4[j];
			if(k==0) continue;
			kk=FindNode(a,nodes,k);
			if(kk==0) continue;
			if(a[kk]->final_thr < thr1) continue;
			a[kk]->final_thr = thr1;
		}
	}

/* backup ID list and recover ID list */  
	for(i=1;i<=nodes;i++){
		if(!a[i]->active) continue;		/* 308 */
	/* final_thr remains initialized. */
	/* last_thr is the largest of the given thr values */
		if(a[i]->final_thr > last_thr){
			backupIDlist(a,nodes,i);
			clearIDlist(a,nodes,i);
		}
	}

/* remove the items from all other lists, if present at a higher thr level. */
	for(i=1;i<=nodes;i++){
		if(!a[i]->active) continue;		/* 308 */
		if(a[i]->final_thr != thr1 || a[i]->n4==0) continue;
	
		for(j=0;j<a[i]->n4;j++){
			k=a[i]->idlist4[j];
			if(k==0) continue;
			kk=FindNode(a,nodes,k);
			if(kk==0) continue;
			if(a[kk]->final_thr < thr) continue;
			for(m=1;m<=nodes;m++){
				if(a[m]->final_thr < thr || !a[m]->active) continue;	/* 308 */
				if(a[m]->n4 == 0) continue;
				for(n=0;n<a[m]->n4b && a[m]->n4b>1;n++){
					if(a[m]->idlist4b[n] == k){
						last=a[m]->n4b - 1;
						if(n<last) a[m]->idlist4b[n]=a[m]->idlist4b[last];
						a[m]->idlist4b[last]=0;
						a[m]->n4b -= 1;
						if(n == last) break;
						n--;
					}
				}
			}
		}
	}

	for(i=1;i<=nodes;i++){
		if(!a[i]->active) continue;		/* 308 */
		if(a[i]->final_thr == thr1){
			recoverIDlist(a,nodes,i);
		}
	}

	return FALSE;
}
/***************************************************/


/***************************************************/
unsigned CountBits(unsigned long ID)
/* This returns the number of non-zero bits. */
/***************************************************/
{
	unsigned i=0;

	while(ID!=0){
		if(ID & 0x01) i += 1;
		ID >>= 1;
	}
	
	return i;
}
/***************************************************/


/***************************************************/
unsigned ShowHighestBit(unsigned long ID)
/* This returns the highest bit. */
/***************************************************/
{
	unsigned i=0;

	while(ID!=0){
		ID >>= 1;
		i += 1;
	}
	
	return i;
}
/***************************************************/


/************************************************************/
void ReassignDomainID(Node **a,unsigned nodes,unsigned i)
/************************************************************/
{
	unsigned j,k,m,n;
	unsigned DMnum = a[i]->DMnum;
	unsigned counter = 1;
	unsigned cmax = 0;		/* maximum value for counter */
	Boolean flag = FALSE;
	unsigned largest,temp,next;
	unsigned long oldID,newID;

	for(counter=1;counter<=DMnum;counter++){
		for(j=0;j<DMnum;j++){
			if(a[i]->dmlist[j].nodeID == a[i]->n0){
				if(ShowHighestBit(a[i]->dmlist[j].ID) == counter){
					cmax +=1;
					break;
				}
			}
		}
	}

	for(counter=1;counter<=cmax;counter++){
		flag = FALSE;
		for(j=0;j<DMnum;j++){
			if(a[i]->dmlist[j].nodeID == a[i]->n0){
				if(ShowHighestBit(a[i]->dmlist[j].ID) == counter){
					flag = TRUE;
					break;
				}
			}
		}
		if(flag) continue;
		else{
			largest = 1;
			for(k=0;k<DMnum;k++){
				if(a[i]->dmlist[k].nodeID == a[i]->n0){
					if((temp=ShowHighestBit(a[i]->dmlist[k].ID)) > largest){
							largest = temp;
					}
				}
			}

			next = largest;
			for(k=0;k<DMnum;k++){
				if(a[i]->dmlist[k].nodeID == a[i]->n0){
					if((temp=ShowHighestBit(a[i]->dmlist[k].ID)) > counter) {
						if(temp < next){
							next = temp;
						}
					}
				}
			}

/* for debug */
			printf("Reassigning a[%u]->dmlist, nodeID=%u\n",\
				i,a[i]->n0);
			oldID = bit2[next];
			newID = bit2[counter];
			if(oldID <= newID) continue;	/* This should not happen. */

			for(k=0;k<DMnum;k++){
				if(a[i]->dmlist[k].nodeID == a[i]->n0){
					if(a[i]->dmlist[k].ID & oldID){
						a[i]->dmlist[k].ID ^= oldID;
						a[i]->dmlist[k].ID |= newID;
/* for debug */
						printf("New value\ta[%u]->dmlist[%u].ID=%lu, oldID=%lu, newID=%lu.\n",\
							i,k,a[i]->dmlist[k].ID,oldID,newID);
					}
				}
			}



/* change all IDs according to the new assignment */
			for(m=1;m<=nodes;m++){				/* corrected in 308 */
				for(n=0;n<a[m]->DMnum;n++){
					if(a[m]->dmlist[n].nodeID == a[i]->n0 && (a[m]->dmlist[n].ID & oldID)){
						a[m]->dmlist[n].ID ^= oldID;
						a[m]->dmlist[n].ID |= newID;
					}
				}
			}
		}
	}
	a[i]->DMmax = cmax;
		
	return;
}
/************************************************************/

/************************************************************/
DMlist *expandDMlist(DMlist *dmlist_in,unsigned n,unsigned m)
/* n is the total number of DMlist, m is the current 
working point. 
Now no limitation is imposed on n. However, the domain ID
is limited to 30bit. */
/************************************************************/
{
	unsigned i;

	if(n>50){
		fprintf(stderr,"Number of domains larger than 50.\n");
		error_expandDMlist = TRUE;
		return dmlist_in;
	}
		
	if((dmlist_in=(DMlist *)realloc(dmlist_in,(n+1)*sizeof(DMlist)))==NULL){
		fprintf(stderr,"Error in allocating memory in expandDMlist1.\n");
		exit(1);
	}
	for(i=n;i>m;i--){
		dmlist_in[i].nodeID = dmlist_in[i-1].nodeID;
		dmlist_in[i].start = dmlist_in[i-1].start;
		dmlist_in[i].end = dmlist_in[i-1].end;
		if(i>m+1) dmlist_in[i].ID = dmlist_in[i-1].ID;
		else dmlist_in[i].ID = 0;	/* This will be set later. */
/*		dmlist_in[i].ID = bit2[i+1];
*/
	}

	return dmlist_in;
}
/************************************************************/


/************************************************************/
void SearchMultiDomain(Node **a,unsigned nodes,unsigned i)
/* domain ID is expressed as 1,2,4,8,16,32,64,128, or
0x01, 0x02, 0x04, 0x08, 0x10, 0x11, 0x12, 0x14, 0x18.
These represent ID numbers, 1,2,3,4,5,6,7,and 8.
If 8 different domains are present, 0xFF. Maximum number of
domains is limited to 30.*/
/* DMlist is initalized in main(). DMnum = 1 before start. */
/* DMmax is used to hold the maximum number of domain ID.
   Homologous domains within a protein is given an
   identical domain ID. */
/************************************************************/
{
	unsigned j,m,n,jj,mm,k1,k2,k3,k4;
	unsigned n3=a[i]->n3;
	unsigned n4=a[i]->n4;
	unsigned start,end;
	unsigned start2,end2;
	unsigned a1,a2,b1,b2;
	unsigned c1,c2,d1,d2;
	double score;
	double level1 = _level1 * 0.9;
	double level3 = _level3;
	double level4 = _level4;
	unsigned min_domain_size = _min_domain_size;
	unsigned min_domain_size2 = min_domain_size / 10;
	DMlr *dmlr_1;
	Boolean flag = FALSE;
	unsigned DMmax;
	static SQlist tempSQ;

	if(n3 < 2) return;

	a1=a2=b1=b2=0;

	/* Defining domains */

	printf("Defining domains for i=%u.\n",i);

	/* initial assignment of domains. for each of idlist4. */
	for(j=0;j<n4;j++){
		k1=a[i]->idlist4[j];
		k2=FindNode(a,nodes,k1);
		if(k2==0) continue;
		for(jj=0;jj<n3;jj++){
			if(a[i]->sqlist3[jj].qID == k1){
				b1=a[i]->sqlist3[jj].Qstart;
				b2=a[i]->sqlist3[jj].Qend;
			} else if(a[i]->sqlist3[jj].ID == k1){
				b1=a[i]->sqlist3[jj].Sstart;
				b2=a[i]->sqlist3[jj].Send;
			} else continue;

			for(m=0;m<a[k2]->DMnum;m++){
				a1=a[k2]->dmlist[m].start;
				a2=a[k2]->dmlist[m].end;
				score = OverlapScore(a1,a2,b1,b2);
				if(score > level1) break;
				else if(level3 < score && score <= level1){ 
					dmlr_1 = OverlapDomains(a1,a2,b1,b2);
					if(dmlr_1->overlap <= min_domain_size) continue;
/*
printf("Ar =%u, Al =%u, Br =%u, Bl =%u, overlap =%u.\n",dmlr_1->Ar,dmlr_1->Al, dmlr_1->Br,dmlr_1->Bl,dmlr_1->overlap);
*/				

					if(dmlr_1->Al > min_domain_size){
						a[k2]->dmlist=expandDMlist(a[k2]->dmlist,a[k2]->DMnum++,m);
						if(error_expandDMlist){
							a[k2]->DMnum--;
							printf("i=%u, %s, k2=%u, %s.\n",i, a[i]->name,k2,a[k2]->name);
							PrintDomain(stdout,a,nodes,k2);
							exit(1);
						}
						a[k2]->dmlist[m+1].start=b1;
						a[k2]->dmlist[m].end=b1-1;
						DMmax = a[k2]->DMmax++;
						a[k2]->dmlist[m+1].ID = bit2[DMmax+1];
						m++;
					}
					if(dmlr_1->Ar > min_domain_size){
						a[k2]->dmlist=expandDMlist(a[k2]->dmlist,a[k2]->DMnum++,m);
						if(error_expandDMlist){
							a[k2]->DMnum--;
							printf("i=%u, %s, k2=%u, %s.\n",i, a[i]->name,k2,a[k2]->name);
							PrintDomain(stdout,a,nodes,k2);
							exit(1);
						}
						a[k2]->dmlist[m+1].start=b2+1;
						a[k2]->dmlist[m].end=b2;
						DMmax = a[k2]->DMmax++;
						a[k2]->dmlist[m+1].ID = bit2[DMmax+1];
						m++;
					}
					if(a2 < b2) b1 = a2 + 1;	/* for the next cycle */
					break;
				}
			}		/* m */
		}		/* jj */
	}



	/* writing direct and indirect homologous domains */

	for(j=0;j<n4;j++){
		k1=a[i]->idlist4[j];
		k2=FindNode(a,nodes,k1);
		if(k2==0) continue;
		for(jj=0;jj<n3;jj++){
			if(a[i]->sqlist3[jj].qID == k1){
				copySQlist(&tempSQ,&a[i]->sqlist3[jj]);
			} else if(a[i]->sqlist3[jj].ID == k1){
				RcopySQlist(&tempSQ,&a[i]->sqlist3[jj]);
			} else continue;

			b1=tempSQ.Qstart;
			b2=tempSQ.Qend;
			k3=FindNode(a,nodes,tempSQ.ID);
			if(k2==k3) continue;

/* this may not be removed 
*/
			for(m=0;m<j;m++){
				if(a[k3]->n0 == a[i]->idlist4[m]) break;
			}
			if(m<j) continue;

			flag = FALSE;
			for(m=0;m<a[k2]->DMnum;m++){
				a1 = start = a[k2]->dmlist[m].start;
				if(start > min_domain_size2) start -= min_domain_size2;
				else start = 1;
				start2 = a1 + min_domain_size2;
				a2 = a[k2]->dmlist[m].end;
				end = a2 + min_domain_size2;
				if(end > a[k2]->len) end = a[k2]->len;
				end2 = a2 - min_domain_size2;

				if(!flag && (start <= b1 && b1 <= end2)) flag = TRUE;
				if(flag && (b2 < start2)){
					flag = FALSE;
					break;
				}
				if(flag){

				/* the coordinates for k2 are converted to the coordinates for k3 */

					if(tempSQ.Sstart + a1 > b1) c1=tempSQ.Sstart + a1 - b1;
					else c1=1;
					c2=tempSQ.Sstart + a2 - b1;

					for(mm=0;mm<a[k3]->DMnum;mm++){
						d1=a[k3]->dmlist[mm].start;
						d2=a[k3]->dmlist[mm].end;
						score=OverlapScore(d1,d2,c1,c2);

/*
printf("d1=%u, d2=%u, c1=%u, c2=%u, score=%f\n",d1,d2,c1,c2,score);
*/

						if(score <= level4) continue;
						for(n=0;n<=j;n++){
							if(a[k2]->dmlist[m].nodeID == a[i]->idlist4[n]) break;
						}
						if(n<=j){
							k4=n;
							for(n=0;n<k4;n++){
								if(a[k3]->dmlist[mm].nodeID == a[i]->idlist4[n]) break;
							}
							if(n<k4){ 
								if(score > level1){		/* nearly overlapping */
									if(!isMultiDomain(a[k3]->dmlist[mm].ID)){
										if(a[k2]->dmlist[m].nodeID == a[k3]->dmlist[mm].nodeID){
											a[k2]->dmlist[m].ID |= a[k3]->dmlist[mm].ID;
										} else {
											a[k2]->dmlist[m].nodeID = a[k3]->dmlist[mm].nodeID;
											a[k2]->dmlist[m].ID = a[k3]->dmlist[mm].ID;
										}
										break;
									}
								} else {
									continue;
								}
							}
						}

						/* k2 comes before k3 in idlist4 */

						if(isMultiDomain(a[k2]->dmlist[m].ID)) continue;
						if(a[k3]->dmlist[mm].nodeID == a[k2]->dmlist[m].nodeID){
							a[k3]->dmlist[mm].ID |= a[k2]->dmlist[m].ID;
							continue;
						} else {
							dmlr_1 = OverlapDomains(d1,d2,c1,c2);
							if(dmlr_1->overlap <= min_domain_size) continue;

/*							
							if(dmlr_1->Al > min_domain_size){
								a[k3]->dmlist=expandDMlist(a[k3]->dmlist,a[k3]->DMnum++,mm);
								if(error_expandDMlist){
									a[k3]->DMnum--;
									printf("i=%u, %s, k3=%u, %s.\n",i, a[i]->name,k3,a[k3]->name);
									PrintDomain(stdout,a,nodes,k3);
									exit(1);
								}
								if(a[i]->sqlist3[jj].qID == k1){
									a[k3]->dmlist[mm+1].start=d1 + a[i]->sqlist3[jj].Sstart - b1;
								} else {
									a[k3]->dmlist[mm+1].start=d1 + a[i]->sqlist3[jj].Qstart - b1;
								}
								a[k3]->dmlist[mm].end=a[k3]->dmlist[mm+1].start - 1;
								DMmax = a[k3]->DMmax++;
								a[k3]->dmlist[mm].ID = bit2[DMmax+1];
								mm++;
							}
*/

							a[k3]->dmlist[mm].nodeID = a[k2]->dmlist[m].nodeID;
							a[k3]->dmlist[mm].ID = a[k2]->dmlist[m].ID;


/*
							if(dmlr_1->Br > min_domain_size){
								a[k3]->dmlist=expandDMlist(a[k3]->dmlist,a[k3]->DMnum++,mm);
								if(error_expandDMlist){
									a[k3]->DMnum--;
									printf("i=%u, %s, k3=%u, %s.\n",i, a[i]->name,k3,a[k3]->name);
									PrintDomain(stdout,a,nodes,k3);
									exit(1);
								}
								if(a[i]->sqlist3[jj].qID == k1){
									a[k3]->dmlist[mm].end=d2 + a[i]->sqlist3[jj].Sstart - b1;
								} else {
									a[k3]->dmlist[mm].end=d2 + a[i]->sqlist3[jj].Qstart -b1;
								}
								a[k3]->dmlist[mm+1].start=a[k3]->dmlist[mm].end + 1;
								DMmax = a[k3]->DMmax++;
								a[k3]->dmlist[mm+1].ID = bit2[DMmax+1];
								mm++;
							}
*/

							break;
						}


					}		/* mm */
				} 
			}		/* m */

		}		/* jj */
	}		/* j */


	for(j=0;j<n4;j++){
		k1=a[i]->idlist4[j];
		k2=FindNode(a,nodes,k1);
		if(k2==0) continue;
		ReassignDomainID(a,nodes,k2);
	}

	PrintDomain(stdout,a,nodes,i);
	fflush(stdout);
	
	return;
}
/***************************************************/

/***************************************************/
void CopyStack(Stack *stack1,Stack *stack2)
/***************************************************/
{
	unsigned m;

	stack1->nodeID = stack2->nodeID;
	stack1->ID = stack2->ID;
	stack1->count = stack2->count;
	for(m=0;m<40;m++) stack1->IDcount[m] = stack2->IDcount[m];

	return;
}
/***************************************************/


/***************************************************/
void InitStack(Stack *stack1,unsigned size)
/***************************************************/
{
	unsigned n,m;

	for(n=0;n<size;n++){
		stack1[n].nodeID = 0;
		stack1[n].ID = 0L;
		stack1[n].count = 0;
		for(m=0;m<40;m++) stack1[n].IDcount[m]=0;
	}

	return;
}
/***************************************************/


/***************************************************/
int RemoveMultiDomain(Node **a,unsigned nodes,unsigned i)

/* New algorithm.
1. Make a histogram of domains.
2. Classify sequences according to domain composition.
3. Split idlist4.
4. Split sqlist3 and mark crosslink in sqlist3.
*/
/***************************************************/
{
	unsigned j,k,m,n,jj,p;
	unsigned p1,p2,max_p1,max_p2;
	unsigned n3=a[i]->n3;
	unsigned n4=a[i]->n4;

	static Stack *max_stack;
	unsigned highest = HIGHEST;	/* highest size of max_count */
	unsigned counter,counter1,counter2,max_counter,max_counter1,max_counter2;
	double cutoff_level[5]={0.8, 0.85, 0.9, 0.95, 1.0};
	unsigned cutoff_count;
	unsigned cutoff_count_save[5];
	unsigned count_save[5];
	static unsigned max_n4=0;
	static Boolean *flag;

	unsigned last;
	Boolean md_present = FALSE;
	static Stack *stack1;
	unsigned num;
	unsigned long ID1,ID2,tempID;
	
	if(n4 < 2) return FALSE;

	if(!a[i]->active) return FALSE;		/* 308 */


	for(j=0;j<n4-1;j++){
		for(m=j+1;m<n4;m++){
			if(a[i]->idlist4[j]==a[i]->idlist4[m]){
				swapUnsigned(&a[i]->idlist4[m],&a[i]->idlist4[(n4--) - 1]);
			}
		}
	}
	a[i]->n4 = n4;

	if(stack1==NULL){
		stack1 = (Stack*)calloc(_stack_size + 1, sizeof(Stack));
	} else {
		InitStack(stack1,_stack_size);
	}
	if(max_stack==NULL){
		max_stack = (Stack*)calloc(21, sizeof(Stack));
	} else {
		InitStack(max_stack,20);
	}
	if(max_n4 == 0) max_n4 = 1;
	if(flag==NULL){
		flag = (Boolean*)calloc(max_n4,20*sizeof(Boolean));
	}
	if(max_n4 < n4 + 1){
		max_n4 = n4 + 1;
		flag = (Boolean*)realloc(flag,max_n4*20*sizeof(Boolean));
	}
	

/* check of nodeID */
	md_present=FALSE;

	/* making histogram of nodeID */
	/* one nodeID is counted once per dmlist */
	num=0;
	for(j=0;j<n4;j++){
		for(n=0;n<num;n++){
			if(stack1[n].count < 0) stack1[n].count = 0 - stack1[n].count;
		}
		k = FindNode(a,nodes,a[i]->idlist4[j]);

		for(m=0;m<a[k]->DMnum;m++){
			for(n=0;n<num;n++){
				if((stack1[n].nodeID == a[k]->dmlist[m].nodeID)){
					for(p=0;p<32;p++){
						if(bit2[p] & a[k]->dmlist[m].ID) stack1[n].IDcount[p] += 1;
					}
					if(stack1[n].count >= 0){
						stack1[n].ID |= a[k]->dmlist[m].ID;
						stack1[n].count = -1 - stack1[n].count;	/* to mark that this item has been counted. */
					} else {
						stack1[n].ID |= a[k]->dmlist[m].ID;
					}
					break;
				}
			}
			if(n==num){
				stack1[num++].nodeID=a[k]->dmlist[m].nodeID;
				for(p=0;p<32;p++){
					if(bit2[p] & a[k]->dmlist[m].ID) stack1[n].IDcount[p] += 1;
				}
				stack1[n].count = -1;
				stack1[n].ID |= a[k]->dmlist[m].ID;
			}
		}
	}
	for(n=0;n<num;n++){
		if(stack1[n].count < 0) stack1[n].count = 0 - stack1[n].count;
	}

	/* selecting most abundant nodeID */
	if(highest > 18) highest = 18;

	for(n=0;n<num;n++){
		for(j=0;j<highest;j++){
			if(max_stack[j].count < stack1[n].count){
				for(jj=highest-1;jj>j;jj--){
					max_stack[jj].nodeID = max_stack[jj-1].nodeID;
					max_stack[jj].count = max_stack[jj-1].count;
					max_stack[jj].ID = max_stack[jj-1].ID;
					for(p=0;p<32;p++) max_stack[jj].IDcount[p] = max_stack[jj-1].IDcount[p];
				}
				max_stack[j].nodeID = stack1[n].nodeID;
				max_stack[j].count = stack1[n].count;
				max_stack[j].ID = stack1[n].ID;
				for(p=0;p<32;p++) max_stack[jj].IDcount[p] = stack1[n].IDcount[p];
				break;
			}
		}
	}

/* for debug */
	for(n=0;n<num;n++){
		printf("n=%u, stack1[n].nodeID=%u, stack1[n].count=%u\n",n,stack1[n].nodeID,stack1[n].count);
	}
	printf("n4=%u\t",n4);
	printf("max_stack[0].nodeID=%u, max_stack[1].nodeID=%u, max_stack[0].count=%u, max_stack[1].count=%u\n",\
		max_stack[0].nodeID, max_stack[1].nodeID, max_stack[0].count, max_stack[1].count); 
/* end debug */

	/* search domain ID */
	for(n=0;n<5;n++){
		cutoff_count_save[n] = 0;
		count_save[n] = max_stack[n].count;
	}
	
	for(n=0;n<highest;n++){
		if(max_stack[n].count < 0.1 * n4){
			max_stack[n].ID = 0x0;
			continue;
		}
		for(cutoff_count=0;cutoff_count<4;cutoff_count++){
			max_stack[n].ID = 0x0;
			for(p=0;p<32;p++){
				if(max_stack[n].IDcount[p] > cutoff_level[cutoff_count] * max_stack[n].count){
					max_stack[n].ID |= bit2[p];
				}
			}
			counter=0;
			for(j=0;j<n4;j++){
				k = FindNode(a,nodes,a[i]->idlist4[j]);
				ID1=0x0;
				for(m=0;m<a[k]->DMnum;m++){
					if((max_stack[n].nodeID == a[k]->dmlist[m].nodeID)){
						ID1 |= a[k]->dmlist[m].ID;
					}
				}
				if((max_stack[n].ID & ID1) == max_stack[n].ID) counter += 1;
			}
			/* for debug */
			if(max_stack[n].ID){
				printf("max_stack[%u].nodeID=%u, ID=%s, count=%u\n",\
				n,max_stack[n].nodeID,c10_2(max_stack[n].ID,32),counter);
			}
			if(counter > 0.7 * max_stack[n].count || max_stack[n].ID == 0x1){
				max_stack[n].count = counter;
				cutoff_count_save[n] = cutoff_count;
				break;
			}
		}
		if(cutoff_count==4) max_stack[n].ID=0x0;
	}

	/* find two major bits that are most widely shared with minimal overlap */
	for(n=0;n<highest;n++){
		if(max_stack[n].count >= 0.1 * n4 && max_stack[n].ID == 0x0){
			max_p1 = max_p2 =0;
			max_counter = 0;
			max_counter1 = max_counter2 = 0;
			for(p1=1;p1<31;p1++){
				for(p2=p1+1;p2<32;p2++){
					counter1 = counter2 = 0;
					for(j=0;j<n4;j++){
						k = FindNode(a,nodes,a[i]->idlist4[j]);
						ID2=0x0;
						for(m=0;m<a[k]->DMnum;m++){
							if((max_stack[n].nodeID == a[k]->dmlist[m].nodeID)){
								ID2 |= a[k]->dmlist[m].ID;
							}
						}
						if((bit2[p1] & ID2) && ((bit2[p2] & ID2) == 0x0)) counter1 += 1;
						if(((bit2[p1] & ID2) == 0x0) && (bit2[p2] & ID2)) counter2 += 1;
					}
					if(max_counter < counter1 + counter2){
						max_counter = counter1 + counter2;
						max_counter1 = counter1;
						max_counter2 = counter2;
						max_p1 = p1;
						max_p2 = p2;
					}
				}
			}
			if(max_counter > 0.8 * max_stack[n].count){
				for(j=highest;j>n;j--){
					CopyStack(&max_stack[j],&max_stack[j-1]);
					if(j<5) count_save[j] = count_save[j-1];
				}
				max_stack[n].ID = bit2[max_p1];
				max_stack[n+1].ID = bit2[max_p2];
				max_stack[n].count = max_counter1;
				max_stack[n+1].count = max_counter2;

			/* for debug */
				printf("stack has been rebuilt.\n");
				for(j=0;j<highest;j++){
					if(max_stack[j].ID){
						printf("max_stack[%u].nodeID=%u, ID=%s, count=%u\n",\
						j,max_stack[j].nodeID,c10_2(max_stack[j].ID,32),max_stack[j].count);
					}
				}
			}
		}
	}


/* size of top three stacks */
	counter=0;
	for(n=0;n<3;n++){
		counter += max_stack[n].count;
	}
	if(counter < 0.95 * n4){

		/* search domain ID again */
		for(n=0;n<3;n++){
			if(count_save[n] < 0.1 * n4){
				continue;
			}
			for(cutoff_count=cutoff_count_save[n]+1;cutoff_count<4;cutoff_count++){
				tempID = 0x0;
				for(p=0;p<32;p++){
					if(max_stack[n].IDcount[p] > cutoff_level[cutoff_count] * count_save[n]){
						tempID |= bit2[p];
					}
				}
				counter=0;
				for(j=0;j<n4;j++){
					k = FindNode(a,nodes,a[i]->idlist4[j]);
					ID1=0x0;
					for(m=0;m<a[k]->DMnum;m++){
						if((max_stack[n].nodeID == a[k]->dmlist[m].nodeID)){
							ID1 |= a[k]->dmlist[m].ID;
						}
					}
					if((tempID & ID1) == tempID) counter += 1;
				}
				/* for debug */
				if(tempID){
					printf("max_stack[%u].nodeID=%u, tempID=%s, count=%u\n",\
					n,max_stack[n].nodeID,c10_2(tempID,32),counter);
				}
				if(counter > count_save[n] || tempID == 0x1){
					max_stack[n].count = counter;
					max_stack[n].ID = tempID;
					break;
				}
			}
			/* if(cutoff_count==4) continue; */
		}
	}




	/* Search adequate set of stacks */
	for(m=0;m<n4;m++){
		for(n=0;n<20;n++){
			flag[20*m+n] = FALSE;
		}
	}
	counter = counter1 = counter2 = 0;
	for(j=0;j<n4;j++){
		k = FindNode(a,nodes,a[i]->idlist4[j]);
		if(k>nodes || k==0) continue;
		for(n=0;n<highest;n++){
			if(max_stack[n].ID==0) continue;
			ID1 = 0x0;
			for(m=0;m<a[k]->DMnum;m++){
				if(max_stack[n].nodeID==a[k]->dmlist[m].nodeID){
					ID1 |= a[k]->dmlist[m].ID;
				}
			}
			if((ID1 & max_stack[n].ID) == max_stack[n].ID){
				flag[20*j+n]=TRUE;
			}
		}
	}

	max_counter1 = max_counter2 = 0;
	max_p1 = 0;
	max_p2 = 1;
	for(m=0;m<highest-1;m++){
		for(n=m+1;n<highest;n++){
			counter1 = counter2 = 0;
			for(j=0;j<n4;j++){
				if(flag[20*j+m] && flag[20*j+n]) counter1 += 1;
				if(flag[20*j+m] || flag[20*j+n]) counter2 += 1;
			}
			if((max_counter1 == 0 || max_counter1 > 2)){
				if((0 == counter1 || counter1 > 2) && (max_counter2 - max_counter1 >= counter2 - counter1)) continue; 
				else if((0 < counter1 && counter1 < 3) && ((max_counter2 - max_counter1) * 0.7 >= (counter2 - counter1))) continue;
			}else{
				if(0 == counter1 || counter1 > 2) continue;
				else if((0 < counter1 && counter1 < 3) && (max_counter2 - max_counter1 >= counter2 - counter1)) continue;
			}
			max_counter1 = counter1;
			max_counter2 = counter2;
			max_p1 = m;
			max_p2 = n;
		}
	}
	printf("max_p1=%u, nodeID=%u, max_p2=%u, nodeID=%u.\n",max_p1, max_stack[max_p1].nodeID,max_p2,max_stack[max_p2].nodeID);

	if(max_p2 == 0) return FALSE;
	else if(max_stack[max_p2].count < 3) return FALSE;
	else if(max_counter2 - max_counter1 < 0.5 * n4) return FALSE;

	/* removing multidomain from idlist4 */
	for(j=0;j<n4;j++){
		k = FindNode(a,nodes,a[i]->idlist4[j]);
		if(k>nodes || k==0) continue;
		if(flag[20*j+max_p1] && flag[20*j+max_p2]){
			n = a[k]->n0;
			printf("j=%u, Node %u: %s is being identified as multidomain.\n",j,n,a[k]->name);
			fflush(stdout);
			md_present = TRUE;
			a[i]->idlist4[j] += nodes;	/* this is to mark idlist */
			if(a[k]->pID == 0) a[k]->pID = a[i]->n0;
			SetMultiDomain(a,nodes,k);
		}
	}

	if(!md_present) return FALSE;

/* packing idlist4 by cleaning removed items */
	current_n4=n4;	/* static variable */
	current_n4b=a[i]->n4b;	/* static variable */
	last = n3;
	for(j=0;j<current_n4;j++){
		if(a[i]->idlist4[j] > nodes){

			k=a[i]->idlist4[j] - nodes;

			/* revert dmlist by eliminating nodeID of multidomain */
			for(jj=0;jj<current_n4;jj++){
				n=FindNode(a,nodes,a[i]->idlist4[jj]);
				for(m=0;m<a[n]->DMnum;m++){
					if(a[n]->dmlist[m].nodeID == k){
						a[n]->dmlist[m].nodeID = a[n]->n0;
						a[n]->dmlist[m].ID = bit2[1+a[n]->DMmax++];
					}
				}
			}

			/* eliminate multidomain from idlist4, idlist4b  and sqlist3 */
			a[i]->idlist4[j] = 0;
			swapUnsigned(&a[i]->idlist4[j],&a[i]->idlist4[(current_n4--) - 1]);

/* remove nodeID from others */
			for(m=0;m<current_n4b;m++){
				if(a[i]->idlist4b[m] == a[i]->idlist4[j]){
					swapUnsigned(&a[i]->idlist4b[m],&a[i]->idlist4b[current_n4b-- - 1]);
					m--;
				}
			}
					
			for(m=0;m<last;m++){
				if(k==a[i]->sqlist3[m].ID || k==a[i]->sqlist3[m].qID){
					swapSQlist(&a[i]->sqlist3[m],&a[i]->sqlist3[last-- - 1]);
					m--;
				}
			}
			j--;
		}
	}
	a[i]->n3 = last;
	a[i]->n4 = current_n4;	
	a[i]->n4b = current_n4b;	

/* The groups containing multidomain are removed. Now they are classified. */




	
	return md_present;
}
/***************************************************/

/******************************************************/
void SetMultiDomain(Node **a,unsigned nodes,unsigned n)
/******************************************************/
{
	if(n==0 || n>nodes) return;

	a[n]->domain |= 0x1;
	a[n]->n6 = 1;
	a[n]->n4 = 1;
	a[n]->n4b = 1;
	a[n]->final_thr = thr;
	a[n]->idlist4[0] = a[n]->n0;
	a[n]->sublist6[0] = a[n]->n0;

	return;
}
/******************************************************/


/***************************************************/
int RegionTest_idlist4(Node **a,unsigned nodes,unsigned i)
/***************************************************/
{

	return TRUE;
}
/***************************************************/



/***************************************************/
void recoverIDlist(Node **a,unsigned nodes,unsigned i)
/***************************************************/
{
	unsigned j;
	unsigned n4=a[i]->n4;
	unsigned n4b=a[i]->n4b;

	if(n4b==0) return;
	if(i > nodes) return;

	if(n4 < n4b){
		if(n4==0){
			if((a[i]->idlist4=(unsigned*)calloc(n4b+2,sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in recoverIDlist1.\n");
				fprintf(stderr,"i=%u, n4=%u, n4b=%u.\n",i,n4,n4b);
				exit(1);
			}
		} else {
			if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n4b+2)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in recoverIDlist2.\n");
				fprintf(stderr,"i=%u, n4=%u, n4b=%u.\n",i,n4,n4b);
				exit(1);
			}
		}
		n4=n4b;
	}
	a[i]->n4 = a[i]->n4b;
	for(j=0;j<n4;j++){
		if(j < n4b) a[i]->idlist4[j] = a[i]->idlist4b[j];
		else a[i]->idlist4[j] = 0;
	}
	a[i]->n3_bak = a[i]->n3_old;
	a[i]->n3b_bak = a[i]->n3b_old;
	
	return;
}
/***************************************************/


/***************************************************/
void backupIDlist(Node **a,unsigned nodes,unsigned i)
/***************************************************/
{
	unsigned j;
	unsigned n4=a[i]->n4;
	unsigned n4b=a[i]->n4b;
	unsigned n3=a[i]->n3;
	unsigned n3_old=a[i]->n3_old;

	if(n4==0) return;
	if(i > nodes) return;

	if(n4b > 0 || a[i]->idlist4b!=NULL){
		if(n4b < n4){
			if((a[i]->idlist4b=(unsigned*)realloc(a[i]->idlist4b,(n4+2)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in backupIDlist1.\n");
				fprintf(stderr,"i=%u, n4=%u, n4b=%u.\n",i,n4,n4b);
				exit(1);
			}
		}
	}
	else{
		if((a[i]->idlist4b=(unsigned*)calloc(n4+2,sizeof(unsigned)))==NULL){
			fprintf(stderr,"Memory allocation error in backupIDlist2.\n");
			fprintf(stderr,"i=%u, n4=%u, n4b=%u.\n",i,n4,n4b);
			exit(1);
		}
	}

	a[i]->n4b = a[i]->n4;
	for(j=0;j<n4;j++){
		a[i]->idlist4b[j] = a[i]->idlist4[j];
	}

	if(n3_old > 0 || a[i]->sqlist3b!=NULL){
		if(n3_old < n3){
			if((a[i]->sqlist3b=(SQlist*)realloc(a[i]->sqlist3b,(n3+2)*sizeof(SQlist)))==NULL){
				fprintf(stderr,"Memory allocation error in backupIDlist3.\n");
				fprintf(stderr,"i=%u, n3=%u, n3_old=%u.\n",i,n3,n3_old);
				exit(1);
			}
		}
	}
	else{
		if((a[i]->sqlist3b=(SQlist*)calloc(n3+2,sizeof(SQlist)))==NULL){
			fprintf(stderr,"Memory allocation error in backupIDlist4.\n");
			fprintf(stderr,"i=%u, n3=%u, n3_old=%u.\n",i,n3,n3_old);
			exit(1);
		}
	}

	a[i]->n3_old = a[i]->n3;
	a[i]->n3b_old = a[i]->n3b;
	
	for(j=0;j<n3;j++){
		copySQlist(&a[i]->sqlist3b[j],&a[i]->sqlist3[j]);
	}

	return;
}
/***************************************************/



/***************************************************/
void clearIDlist(Node **a,unsigned nodes,unsigned i)
/* In this subroutine, n4 is not made zero. */
/***************************************************/
{
	unsigned j;
	unsigned n4=a[i]->n4;

	if(n4==0) return;
	if(i==0 || i > nodes) return;

	for(j=0;j<n4;j++){
		a[i]->idlist4[j] = 0;
	}
	
	return;
}
/***************************************************/


/***************************************************/
void MakeIDlist_clique(Node **a,unsigned nodes,unsigned i)

/* This is a new unit cycle of the MakeIDlist.
   From version 351.*/
 /***************************************************/
{
	unsigned j,n3,n7,num,k,m,n;
	int x,y;
	double overlap,score;

	if(!a[i]->active){
		if(a[i]->n3 != 0) return;		/* 308 */
	} else if(!clique_mode && a[i]->final_thr < thr) return;
	else if(a[i]->domain != 0 && (a[i]->domain ^ 4)){	/* 308 */
		return;
	}
	n7 = a[i]->n7;
	n3 = a[i]->n3;
	if(n3==0) n3=a[i]->n3b;
	if(a[i]->n4 > 0){
		if(a[i]->n4 < n7){
			/* bug fix: n7 in place of n3 352f9 */
			if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n7+1)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in MakeIDlist1.\n");
				exit(1);
			}
		}
	} else {
		if((a[i]->idlist4=(unsigned*)calloc(n7+1,sizeof(unsigned)))==NULL){
			fprintf(stderr,"Memory allocation error in MakeIDlist2.\n");
			exit(1);
		}
	}

	m=1;
	a[i]->idlist4[0]=a[i]->n0;
	if(a[i]->domain & 4){
		a[i]->n4 = 1;
		return;
	}
	for(j=0;j<n7;j++){
		num = a[i]->matchlist7[j].ID;
		if(num==0) continue;
		else if(num > nodes) {
			fprintf(stderr,"Negative ID in MakeIDList3.\n");
		}

		score = a[i]->matchlist7[j].score;
		overlap = a[i]->matchlist7[j].overlap;
		if(overlap >= 1.0) x = 10;
		else if(overlap >= 0.9) x = 9;
		else if(overlap >= 0.8) x = 8;
		else if(overlap >= 0.7) x = 7;
		else if(overlap >= 0.6) x = 6;
		else if(overlap >= 0.5) x = 5;
		else if(overlap >= 0.4) x = 4;
		else if(overlap >= 0.3) x = 3;
		else if(overlap >= 0.2) x = 2;
		else if(overlap >= 0.1) x = 1;
		else if(overlap >= 0.0) x = 0;

		y = num_thr;
		for(k=0;k<=num_thr;k++){
			if(score <= thr_list[k]){
				  y	= k;
				  break;
			}
		}

		if(use_org && z[y*11+x] == -1) continue;

		n=FindNode(a,nodes,num);
		if(n==0) continue;
		if(a[n]->domain != 0) continue;
	
		if(m > n7+1){
			fprintf(stderr,"counter m overflow in MakeIDlist4.\n");
			exit(1);
		}
		for(k=0;k<m;k++){
			if(num == a[i]->idlist4[k]) break;
		}
		if(k==m) a[i]->idlist4[m++] = num;
	}
	a[i]->n4 = m;
	a[i]->idlist4[m]=0;
}
/***************************************************/



/***************************************************/
void MakeIDlist_i(Node **a,unsigned nodes,unsigned i)

/* This is a unit cycle of the MakeIDlist.
   From version 350.*/
 /***************************************************/
{
	unsigned j,n3,num,k,m,n;

	if(!a[i]->active){
		if(a[i]->n3 != 0) return;		/* 308 */
	} else if(!clique_mode && a[i]->final_thr < thr) return;
	else if(a[i]->domain != 0 && (a[i]->domain ^ 4)){	/* 308 */
		return;
	}
	n3 = a[i]->n3;
	if(n3==0) n3=a[i]->n3b;
	if(a[i]->n4 > 0){
		if(a[i]->n4 < n3){
			if((a[i]->idlist4=(unsigned*)realloc(a[i]->idlist4,(n3+1)*sizeof(unsigned)))==NULL){
				fprintf(stderr,"Memory allocation error in MakeIDlist1.\n");
				exit(1);
			}
		}
	} else {
		if((a[i]->idlist4=(unsigned*)calloc(n3+1,sizeof(unsigned)))==NULL){
			fprintf(stderr,"Memory allocation error in MakeIDlist2.\n");
			exit(1);
		}
	}

	m=1;
	a[i]->idlist4[0]=a[i]->n0;
	if(a[i]->domain & 4){
		a[i]->n4 = 1;
		return;
	}
	for(j=0;j<n3;j++){
		num = a[i]->sqlist3[j].ID;
		if(num==0) continue;
		else if(num > nodes) {
			fprintf(stderr,"Negative ID in MakeIDList3.\n");
		}
		if(a[i]->sqlist3[j].score >= thr) continue;
		n=FindNode(a,nodes,num);
		if(n==0) continue;
		if(a[n]->domain != 0) continue;
	
		if(m > n3+1){
			fprintf(stderr,"counter m overflow in MakeIDlist4.\n");
			exit(1);
		}
		for(k=0;k<m;k++){
			if(num == a[i]->idlist4[k]) break;
		}
		if(k==m) a[i]->idlist4[m++] = num;
	}
	a[i]->n4 = m;
	a[i]->idlist4[m]=0;
}
/***************************************************/



/***************************************************/
void MakeIDlist(Node **a,unsigned nodes)

/* From version 350, MekeIDlist_i is separated. */
/* This function is called after sorting the nodes.
Hence, a[i]->n0 != i. */
/***************************************************/
{
	unsigned i;

	printf("\nMaking ID list...\n");
	fflush(stdout);
	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);

		MakeIDlist_i(a,nodes,i);
	}
	printf("\nID list completed.\n");
	fflush(stdout);
}
/***************************************************/


/***************************************************/
void InitClist(unsigned nodes)
/***************************************************/
{
	if((ch=(unsigned*)calloc(nodes+2,sizeof(unsigned)))==NULL){
		fprintf(stderr,"Error in allocating memory for ch in InitClist.\n");
		exit(1);
	}
	ClearClist(nodes);

	return;
}
/***************************************************/



/***************************************************************/
void ClearClist(unsigned nodes)
/***************************************************************/
{
	unsigned i;
	
	for(i=0;i<=nodes;i++){
		ch[i]=0;
	}

	return;
}
/***************************************************************/


/***************************************************/
void WriteParent(Node **a,unsigned nodes)
/***************************************************/
{
	unsigned i, j, pID;
	unsigned num;
	unsigned n4;

	printf("\nWriting parent info ...\n");
	for(i=1;i<=nodes;i++){
		PrintProgress(i,10,1000);
		if(a[i]->n3 == 0) continue;
		if(repeat_mode && a[i]->domain & 0x1) continue;	/* pID of multidomain node has been set during the iteration */
		pID=a[i]->n0;
		a[i]->pID=pID;
		if(a[i]->domain & 0x4) continue;
		n4=a[i]->n4;
		for(j=0;j<n4;j++){
			num=FindNode(a,nodes,a[i]->idlist4[j]);
			if(num==0) continue;
			if(repeat_mode && a[num]->domain != 0) continue;
			a[num]->pID = pID;
		}
	}
	return;
}
/***************************************************/



/***************************************************************/
void RenewList6(Node **a,unsigned nodes,unsigned k)
/***************************************************************/
{
	unsigned i,j,n6;

	n6=a[k]->n6;
	if(n6 == 0) return;
	j=0;
	for(i=0;i<n6;i++){
		if(a[k]->sublist6[i] != 0){
			a[k]->sublist6[j++] = a[k]->sublist6[i];
		}
	}
	a[k]->n6 = j;

	return;
}
/***************************************************************/


/* end of file list.c */
