/*******************************************
print.c
GCLUST
This program reads a list file (generated from BLAST result by bl2ls.pl),
and assembles similarity groups. Output is a *.grp file.
ihis is a C version of the lsort2b.pl.
  Copyright Naoki Sato 2002.

Matrix is now int. July 6, 2002.
Added sublist5. July 7, 2002.
Added sublist6. sublist5 changed to reflist5. July 10, 2002.
Phase 1. July 19, 2002.
R,R1,R2 are made global. Aug. 9, 2002.
u is now global. Sept. 8, 2002.
R is now simple pointer. Sept. 9, 2002.
Bug fix for matrix. Sept. 10, 2002.
Bug fix for Unit matrix. Sept. 30, 2002.
Added mode homsub. Oct. 18, 2002.
Reconstruction. April 30, 2003.
PrintDomain2. Nov. 3, 2004.
*******************************************/

#include "defines.h"

void PrintProgress(unsigned i,unsigned x, unsigned y);
void PrintN4(Node **a,unsigned nodes,int it);
void PrintData(Node **a,unsigned nodes,char *listfile,char *table);
void PrintLDB(FILE *fout,Node **a,unsigned nodes);
void PrintMatrix(FILE *fout,Node **a,unsigned nodes,unsigned i,int *m,unsigned d,char *out_mode);
void PrintSubMatrix(FILE *fout,Node **a,unsigned nodes,unsigned i,int *m,unsigned d,char *out_mode);
void PrintMultiDomain(FILE *fout2,Node **a,unsigned nodes);
int PrintSub(Node **a, unsigned nodes, unsigned i);
void PrintTable(FILE *fout,Node **a,unsigned nodes);
void PrintTable3(FILE *fout,Node **a,unsigned nodes);
void PrintVariables2(FILE *fout);
void PrintDomain(FILE *fout,Node **a,unsigned nodes,unsigned i);
void PrintDomain2(FILE *fout,Node **a,unsigned nodes);
void printDomainList(FILE *fout,Node **a,unsigned nodes,unsigned i);


#ifndef __PRINT__
#define __PRINT__
#endif

#include "gclust.h"

/***************************************************************/
void PrintDomain(FILE *fout,Node **a,unsigned nodes,unsigned i)
/***************************************************************/
{
	unsigned j,k1,k2,m;
	unsigned n4=a[i]->n4;

	if(i==0 || i>nodes) return;
	for(j=0;j<n4;j++){
		k1=a[i]->idlist4[j];
		k2=FindNode(a,nodes,k1);
		if(k2==0) continue;
		fprintf(fout,"Node %u, %s, Domain= %d, DMnum=%u\n",a[k2]->n0,a[k2]->name,a[k2]->domain,a[k2]->DMnum);
		for(m=0;m<a[k2]->DMnum;m++){
			fprintf(fout,"\tstart=%5u, end=%5u, nodeID=%6u, ID=%10lu, ID=%s.\n",a[k2]->dmlist[m].start,a[k2]->dmlist[m].end,\
			a[k2]->dmlist[m].nodeID,a[k2]->dmlist[m].ID,c10_2(a[k2]->dmlist[m].ID,32));
		}
	}
	return;
}
/***************************************************************/

/***************************************************************/
void PrintDomain2(FILE *fout,Node **a,unsigned nodes)
/***************************************************************/
{
	unsigned i,m;

	for(i=1;i<=nodes;i++){
		fprintf(fout,"Node %u, %s, DMnum=%u\n",a[i]->n0,a[i]->name,a[i]->DMnum);
		for(m=0;m<a[i]->DMnum;m++){
			fprintf(fout,"\tstart=%5u, end=%5u, nodeID=%6u, ID=%10lu, ID=%s.\n",a[i]->dmlist[m].start,a[i]->dmlist[m].end,\
			a[i]->dmlist[m].nodeID,a[i]->dmlist[m].ID,c10_2(a[i]->dmlist[m].ID,32));
		}
	}
	return;
}
/***************************************************************/



/***************************************************************/
void printDomainList(FILE *fout,Node **a,unsigned nodes,unsigned i)
/***************************************************************/
{
	unsigned m;

	fprintf(fout,"\n            Domain list\n");
	for(m=0;m<a[i]->DMnum;m++){
		fprintf(fout,"                   %3u:  %s  %5u -%5u\n",\
					m+1,c10_2(a[i]->dmlist[m].ID,a[i]->DMnum),\
					a[i]->dmlist[m].start,a[i]->dmlist[m].end);
	}
	fprintf(fout,"\n");
	return;
}
/***************************************************************/



/***************************************************************************/
void PrintProgress(unsigned i,unsigned x, unsigned y)
/* i : item number being processed.
   x : maximum number to be printed without skipping.
   y : interval to be reported.
*/
/***************************************************************************/
{
	static unsigned temp;

	if(i < x) temp = 0;
	if(i >= 10 * y) y *= 10;
	if(i < x){
		printf("%u ",i);
		fflush(stdout);
		return;
 	} else if(i % y == 0){
		printf("%u ",i);
		fflush(stdout);
		temp = i / y;
		return;
	} else if(temp < i / y){
		printf("%u ",i);
		fflush(stdout);
		temp = i / y;
	}
	return;
}
/***************************************************************************/


/***************************************************************************/
void PrintVariables2(FILE *fout)
/* enabled printing of thr */
/***************************************************************************/
{
	char version_string[100];
	char gclust_variables[20];
	int i;

	strcpy(version_string,VERSION);
	strcpy(gclust_variables,GCLUST_VARIABLES);

	fprintf(fout,"%s version %s\n",gclust_variables,version_string);
	fprintf(fout,"%u\t: largeprotein\n",_largeprotein);
	fprintf(fout,"%u\t: max_n3\n",_max_n3);
	fprintf(fout,"%u\t: max_matrix\n",_max_matrix);
	fprintf(fout,"%f\t: level1\n",_level1);
	fprintf(fout,"%f\t: level2\n",_level2);
	fprintf(fout,"%f\t: level3\n",_level3);
	fprintf(fout,"%f\t: level4\n",_level4);
	fprintf(fout,"%f\t: matchlevel\n",_matchlevel);
	fprintf(fout,"%f\t: minlevel\n",_minlevel);
	fprintf(fout,"%f\t: min_filled\n",_min_filled);
	fprintf(fout,"%f\t: min_filled2\n",_min_filled2);
	fprintf(fout,"%u\t: min_zeros\n",_min_zeros);
	fprintf(fout,"%f\t: min_0_occup\n",_min_0_occup);
	fprintf(fout,"%u\t: number_of_genomes\n",number_of_genomes);
	fprintf(fout,"%u\t: suppress_large_matrix\n",suppress_large_matrix);
	fprintf(fout,"%u\t: stack_size\n",_stack_size);
	fprintf(fout,"%u\t: min_domain_size\n",_min_domain_size);
	fprintf(fout,"%f\t: searchbridge_min_triangle (save mode)\n",_searchbridge_min_triangle);
	fprintf(fout,"%u\t: searchbridge_min_n4\tif zero, identical to number of genomes (save mode)\n",_searchbridge_min_n4);
	fprintf(fout,"%f\t: levelF1, parameters in OptimizeOrg\n",_f1);
	fprintf(fout,"%f\t: levelF2\n",_f2);
	fprintf(fout,"%f\t: levelF3\n",_f3);

	fprintf(fout,"%d\t: min_overlap\n",_min_overlap);
	fprintf(fout,"%d\t: delta_diff\n",_delta_diff);
	fprintf(fout,"%d\t: delta_plus\n",_delta_plus);
	fprintf(fout,"%f\t: min_delta_overlap\n",_min_delta_overlap);
	fprintf(fout,"%f\t: org_measure_0\n",_org_measure_0);
	fprintf(fout,"%f\t: org_measure_1\n",_org_measure_1);
	fprintf(fout,"%f\t: allowance_level_0\n",_allowance_level_0);
	fprintf(fout,"%f\t: allowance_level_1\n",_allowance_level_1);
	fprintf(fout,"%f\t: JS_level_0\n",_JS_level_0);
	fprintf(fout,"%f\t: JS_level_1\n",_JS_level_1);
	fprintf(fout,"%f\t: MDR_level_0\n",_MDR_level1_0);
	fprintf(fout,"%f\t: MDR_level_1\n",_MDR_level1_1);
	fprintf(fout,"%f\t: MDR_max_cluster_0\n",_MDR_max_cluster_0);
	fprintf(fout,"%f\t: MDR_max_cluster_1\n",_MDR_max_cluster_1);
	fprintf(fout,"%u\t: MDR_length1\n",_MDR_length1);
	fprintf(fout,"%u\t: MDR_length2\n",_MDR_length2);
	fprintf(fout,"%e\t: MDR_score1\n",_MDR_score1);
	fprintf(fout,"%e\t: MDR_score2\n",_MDR_score2);
	fprintf(fout,"%d\t: MDR_clique_size\n",_MDR_clique_size);
	fprintf(fout,"%f\t: cross_edge_parameter\n",_cross_edge_parameter);
	fprintf(fout,"%f\t: CC_level_2 (CreateCliques)\n",_CC_level_2);
	fprintf(fout,"%f\t: CC_level_3 (CreateCliques)\n",_CC_level_3);
	fprintf(fout,"%e\t: CC_min_thr (CreateCliques)\n",_CC_min_thr);

	fprintf(fout,"%d\t: num_thr\n",num_thr);
	for(i=0;i<num_thr;i++){
		fprintf(fout,"%10.2e\t: thr level %d\n",thr_list[i],i);
	}

	fprintf(fout,"END\n");
}
/***************************************************************************/


/***************************************************************************/
void PrintN4(Node **a,unsigned nodes,int it)
/* This is for debugging */
/***************************************************************************/
{
	unsigned i,j,k;
	FILE *fout;
	char fname[20];

	if(!print_N4_table) return;

	sprintf(fname,"N4out.%d",it);
	if((fout=fopen(fname,"w"))==NULL){
		fprintf(stderr,"Unable to write to file: %s\n",fname);
		return;
	}
	fprintf(fout,"N4 data output\t%u nodes\n",nodes);
	fprintf(fout,"%10.3e\t# Threshold\n",thr);

	for(i=1;i<=nodes;i++){
		fprintf(fout,"Node %u\n",i);
		fprintf(fout,"%s\t%u\t%s\n",a[i]->name,a[i]->len,a[i]->annot);
		fprintf(fout,"Species %d, kingdom %d\n",a[i]->species,a[i]->kingdom);
		fprintf(fout,"N0 %u : original number\n",a[i]->n0);
		fprintf(fout,"N3 %u, N3b %u, N3_old %u, N3_bak %u\n",a[i]->n3,a[i]->n3b,a[i]->n3_old,a[i]->n3_bak);
		fprintf(fout,"N5 %u, N6 %u\n",a[i]->n5,a[i]->n6);
		fprintf(fout,"final_thr=%10.2e\n",a[i]->final_thr);
		fprintf(fout,"domain=%d, pID=%u\n",a[i]->domain,a[i]->pID);
		fprintf(fout,"N4 %u\n",a[i]->n4);
		for(j=0;j<a[i]->n4;j++){
			k=a[i]->idlist4[j];
			fprintf(fout,"a[i]->n0=%u, i=%u\n",\
			k,FindNode(a,nodes,k));
		}
		if(!clique_mode){
			fprintf(fout,"N4b %u\n",a[i]->n4b);
			for(j=0;j<a[i]->n4b;j++){
				k=a[i]->idlist4b[j];
				fprintf(fout,"a[i]->n0=%u, i=%u\n",\
				k,FindNode(a,nodes,k));
			}
		}
		
		fprintf(fout,"\n");
		fflush(fout);
	}
	fprintf(fout,"END\n");
	fflush(fout);
}
/***************************************************/


/***************************************************************************/
void PrintData(Node **a,unsigned nodes,char *listfile,char *table)
/***************************************************************************/
{
	unsigned i,j,m;
	char gclust_data[20];
	SQlist b;
	FILE *fout;
	char version_string[100];


	strcpy(version_string,VERSION);
	strcpy(gclust_data,GCLUST_DATA);

	printf("Writing data file %s ...\n",gclust_data);
	fflush(stdout);

	if((fout=fopen(gclust_data,"w"))==NULL){
		fprintf(stderr,"Unable to write to file: %s\n",gclust_data);
		return;
	}
	fprintf(fout,"%s\t%u\tversion\t%s\n",gclust_data,nodes,version_string);
	fprintf(fout,"%s\t# List file\n",listfile);
	fprintf(fout,"%s\t# Table file\n",table);
	fprintf(fout,"%10.3e\t# Threshold",thr);
	if(ashikiri) fprintf(fout,"\tTapering mode");
	fprintf(fout,"\n");
	fflush(fout);

	for(i=1;i<=nodes;i++){
		fprintf(fout,"Node %u\n",i);
		fprintf(fout,"%s\t%u\t%s\n",a[i]->name,a[i]->len,a[i]->annot);
		fprintf(fout,"N0 %u",a[i]->n0);
		if(a[i]->domain !=0 || _print_all_domains){
			fprintf(fout," Domain %d\n",a[i]->domain);
			fprintf(fout,"maskID %10lu\n",a[i]->maskID);
			fprintf(fout,"DMnum %u\n",a[i]->DMnum);
			for(m=0;m<a[i]->DMnum;m++){
				fprintf(fout,"%5u\t%5u\t%10lu\n",a[i]->dmlist[m].start,a[i]->dmlist[m].end,\
				a[i]->dmlist[m].ID);
			}
			fprintf(fout,"N3 %u\n",a[i]->n3);
			for(j=0;j<a[i]->n3;j++){
				b=a[i]->sqlist3[j];
				fprintf(fout,"%u %u %u %10.2e %u %u %u\n",\
				b.ID, b.Sstart, b.Send, b.score, b.qID, b.Qstart, b.Qend);
			}
			fprintf(fout,"N4 %u\n",a[i]->n4);
			for(j=0;j<a[i]->n4;j++){
				fprintf(fout,"%u\n",a[i]->idlist4[j]);
			}
			
		}
		else fprintf(fout,"\n");

		fprintf(fout,"N1 %u\n",a[i]->n1);
		for(j=0;j<a[i]->n1;j++){
			b=a[i]->sqlist1[j];
			fprintf(fout,"%u %u %u %10.2e %u %u %u\n",\
			b.ID, b.Sstart, b.Send, b.score, b.qID, b.Qstart, b.Qend);
		}
		
		for(j=a[i]->n2-1;j>=1;j--){
			if(a[i]->n2 == 1) break;
			b=a[i]->sqlist2[j];
			if(b.ID==0 && b.Sstart==0 && b.Send==0){
				if(b.qID==0 && b.Qstart==0 && b.Qend==0){
					a[i]->n2 -= 1;
				}
			}
		}
		fprintf(fout,"N2 %u\n",a[i]->n2);
		for(j=0;j<a[i]->n2;j++){
			b=a[i]->sqlist2[j];
			fprintf(fout,"%u %u %u %10.2e %u %u %u\n",\
			b.ID, b.Sstart, b.Send, b.score, b.qID, b.Qstart, b.Qend);
		}
		fprintf(fout,"\n");
		fflush(fout);
	}
	fprintf(fout,"END\n");
	fflush(fout);
}
/***************************************************/


/***************************************************/
void PrintTable(FILE *fout,Node **a,unsigned nodes)
/***************************************************/
{
	unsigned i;
	char gclust_table2[20];

	strcpy(gclust_table2,GCLUST_TABLE2);

	fprintf(fout,"%s\t%u\n",gclust_table2,nodes);
	
	for(i=1;i<=nodes;i++){
		fprintf(fout,"%s\t%u\t%s\n",a[i]->name,a[i]->len,a[i]->annot);
	}

}
/***************************************************/


/***************************************************/
void PrintTable3(FILE *fout,Node **a,unsigned nodes)
/* This is for gclust3 format */
/***************************************************/
{
	unsigned i;
	char gclust_table3[20];

	strcpy(gclust_table3,GCLUST_TABLE3);

	fprintf(fout,"%s\t%u\n",gclust_table3,nodes);
	
	for(i=1;i<=nodes;i++){
		fprintf(fout,"%u\t%s\t%u\t%s\n",i,a[i]->name,a[i]->len,a[i]->annot);
	}

}
/*****s**********************************************/

/***************************************************/
void PrintLDB(FILE *fout,Node **a,unsigned nodes)
/***************************************************/
{
	unsigned i,n3,n4,j,num,k,m,d;

	for(i=1;i<=nodes;i++){
		n4 = a[i]->n4;
		n3 = a[i]->n3;
		if(n3==0) continue;
		for(j=0;j<n4;j++){
			num = a[i]->idlist4[j];
			m=FindNode(a,nodes,num);
			if(m==0) continue;
			fprintf(fout,"%s %u %s",a[m]->name,a[m]->len,a[m]->annot);
			for(k=0;k < a[m]->n1;k++){
				if(k==0) fprintf(fout,"\t");
				else fprintf(fout," ");
				d=FindNode(a,nodes,a[m]->sqlist1[k].ID);
				fprintf(fout,"%s %8.1e %u-%u",a[d]->name,a[m]->sqlist1[k].score,\
					a[m]->sqlist1[k].Sstart,a[m]->sqlist1[k].Send);
			}
			fprintf(fout,"\n");
		}
		fprintf(fout,"\n");
		if(i%100==0){
			printf("%u ",i);
			fflush(stdout);
		}
	}
	return;
}
/***************************************************/

/*************************************************************************/
void PrintSubMatrix(FILE *fout,Node **a,unsigned nodes,unsigned i,int *m,unsigned d,char *out_mode)
/* matrix is now in int, i.e.,1000*value. */
/*************************************************************************/
{
	unsigned j1,j2,k1,k2;
	unsigned n6 = a[i]->n6;
	unsigned pos;
	float value;

	if(i==0 || i>nodes) return;
	for(j1=0;j1<n6;j1++){
		k1 = FindNode(a,nodes,a[i]->sublist6[j1]);
		if(k1==0) continue;
		fprintf(fout,"%s\t%u\t",a[k1]->name,a[k1]->len);
		for(j2=0;j2<n6;j2++){
			k2 = FindNode(a,nodes,a[i]->sublist6[j2]);
			if(k2==0) continue;
			pos = j1 * d + j2;
			value = (double)m[pos] / 1000.0;
			if(*out_mode=='1'){
				fprintf(fout,"%u",(unsigned)ceil(value));
			} else {
				if( value >= 10.0 ) fprintf(fout,"%4.1f",value);
				else fprintf(fout,"%4.2f",value);
			}
			if(j2<n6-1) fprintf(fout," ");
		}
		fprintf(fout,"\t%s\n",a[k1]->annot);
	}
	fprintf(fout,"\n");
	return;
}
/*************************************************************************/


/*************************************************************************/
void PrintMatrix(FILE *fout,Node **a,unsigned nodes,unsigned i,int *m,unsigned d,char *out_mode)
/* matrix is now in int, i.e.,1000*value. */
/*************************************************************************/
{
	unsigned j1,j2,k1,k2;
	unsigned n4 = a[i]->n4;
	unsigned pos;
	float value;
	int var1,var2;

	if(!a[i]->active) n4 = 1;
	for(j1=0;j1<n4;j1++){
		k1 = FindNode(a,nodes,a[i]->idlist4[j1]);
		fprintf(fout,"%s\t%u\t",a[k1]->name,a[k1]->len);

		if(suppress_large_matrix < n4){
			fprintf(fout," ... ");
		} else {
			for(j2=0;j2<n4;j2++){
				k2 = FindNode(a,nodes,a[i]->idlist4[j2]);
				pos = j1 * d + j2;
				value = (double)m[pos] / 1000.0;
				if(*out_mode == '1'){
					fprintf(fout,"%u",(unsigned)ceil(value));
				} else if(*out_mode == 'r'){
					if( value >= 10.0 ) fprintf(fout,"%4.1f",value);
					else fprintf(fout,"%4.2f",value);
				} else {	/* 355z */
					if(m[pos] == 0){
						fprintf(fout,">1e-3");
					} else {	
						var1 = m[pos] / 100;
						var2 = m[pos] % 100;
						fprintf(fout,"%1de-%d", var2,var1);
					}
				}
				if(j2<n4-1) fprintf(fout," ");
			}
		}

		fprintf(fout,"\t%s",a[k1]->annot);

		if(a[k1]->corr_length != 0 && a[k1]->corr_length < a[k1]->len){
			fprintf(fout," [corr_length=%u]",a[k1]->corr_length);
		}

		fprintf(fout,"\n");

	}
	fprintf(fout,"\n");
	return;
}
/**********************************************/



/***************************************************************/
void PrintMultiDomain(FILE *fout2,Node **a,unsigned nodes)
/* Print items with n6=1. */
/***************************************************************/
{
	unsigned i,j,k,j1,k1;
	char printed=0;

	for(i=1;i<=nodes;i++){
		printed = 0;
		if(a[i]->n3 == 0) continue;

		if(!PrintSub(a,nodes,i)) continue;
		for(j=0;j<a[i]->n4;j++){
			k=FindNode(a,nodes,a[i]->idlist4[j]);
			if(a[k]->n6 != 1) continue;
			printed = 1;
			fprintf(fout2,"Group %u, subgroup %u, %s:\n",a[k]->grpno,a[k]->subgrpno,a[k]->name);
			for(j1=0;j1<a[k]->n1;j1++){
				k1=FindNode(a,nodes,a[k]->sqlist1[j1].ID);
				if(k1==k) continue;
				fprintf(fout2," %u-%u %s,",a[k1]->grpno,a[k1]->subgrpno,a[k1]->name);
			}
			fprintf(fout2,"\n\n");
		}
		if(printed) fprintf(fout2,"\n");
	}
	return;
}
/***************************************************************/

/***************************************************************/
int PrintSub(Node **a, unsigned nodes, unsigned i)
/***************************************************************/
{
	int print_sub = 0;
	unsigned j,k;

	if(no_sub==TRUE) return 0;
	if(a==NULL) return 0;
	if(nodes==0 || i==0) return 0;

	if(a[i]->n5 != 0) print_sub = 1;
	for(j=0;j<a[i]->n4;j++){
		k=FindNode(a,nodes,a[i]->idlist4[j]);
		if(k==0) continue;
		if(a[k]->n6 == a[i]->n5){
			print_sub=0;
			break;
		}
	}
	
	return print_sub;
}
/***************************************************************/

/* end of file print.c */
