/*******************************************
anal.c
GCLUST
previously a part of lsort3.h 
This program reads a list file (generated from BLAST result by bl2ls.pl),
and assembles similarity groups. Output is a *.grp file.
This is a C version of the lsort2b.pl.
  Copyright Naoki Sato 2002.

Added functionality of using catenated list file. June 2, 2002.
Added msort, lnkann. June 16, 2002.
Added SQlist. June 17, 2002.
Added lnkdb. June 25, 2002.
Selection of master entry. June 25, 2002.
Added hom. July 2, 2002.
Matrix is now in float. July 5, 2002.
Matrix is now in int. July 6, 2002.
score is now in float. July 6, 2002. Now double, May 13, 2003.
Added sublist5. July 7, 2002.
Phase 1. July 19, 2002.
RemoveUnrelated returns unsigned. Sept. 5, 2002.
Unnecessary output suppressed. Dec. 30, 2002.
Reconstruction of source code. April 30, 2003.
Modification of makeInfContTable for small proteins. June 29, 2007.
DeltaTest. Oct. 6, 2007.
*******************************************/

#include "defines.h"

int getMinimumY(void);
int selectDeltaFromInf(int delta7,int minimum_y);
int makeInfContTable(int mode);
int MakeDiffTable(void);
double limit_thr(unsigned length);
int thr_level(double thr1);
int CountOrg(int m1,int n1,int mode);
int CountOrgInIDlist(Node **a,unsigned nodes,unsigned i,unsigned j);
int DeltaTest(Node **a, unsigned nodes, unsigned i, unsigned j);
int OptimizeOrg(void);
int select_initial_clique(int m0, int n0, int delta_max);
int search_local_maxima(Boolean print_this_table);
int print_2D_matrix(Node **a,unsigned nodes,unsigned i);
int best_thr(SQlist *s, unsigned n);
void swapSQlist(SQlist *a,SQlist *b);
void swapInt(int *a,int *b);
void swapUnsigned(unsigned *a,unsigned *b);
unsigned overlap(unsigned a1,unsigned a2,unsigned b1,unsigned b2);
int MatchIDlist(Node **a,unsigned nodes,unsigned i, unsigned j);
double OverlapScore(unsigned a1,unsigned a2,unsigned b1,unsigned b2);
DMlr *OverlapDomains(unsigned a1,unsigned a2,unsigned b1,unsigned b2);
int GetScore(Node **a,unsigned nodes,unsigned i, unsigned k);
Region *InitRegion(unsigned i);
void ClearRegion(Region *RR,unsigned sizeRegion);
Region *ReallocRegion(Region *R, unsigned i1, unsigned i2);
double OverlapScore2(SQlist *s,unsigned t2,unsigned len1,unsigned len2);
unsigned GetRegion(SQlist *s,unsigned j2,unsigned d);
unsigned GetRegion2(SQlist *s,unsigned t1,SQlist *t,unsigned t2);
int tTest_idlist4(Node **a,unsigned nodes,unsigned i,unsigned *vector,unsigned dim);
int isMultiDomain(unsigned long domainID);
char *c10_2(unsigned long data, unsigned bits);
char *c10_3(unsigned long data, unsigned long data2, unsigned bits, char x, char y);
unsigned generalEvaluationFunction(unsigned x,double a, double b,double c,double d,double e);
unsigned Evaluation1(unsigned x);
unsigned Evaluation2(unsigned x);

#ifndef __ANAL__
#define __ANAL__
#endif
#include "gclust.h"

double inf[50][6];
/* 0, Smax = logN. N is the number of cells.
 * 1, S = Sigma(pi * logpi).
 * 2, Smax - S.
 * 3, organism number
 * 4, total count
 */

/*************************************/
double limit_thr(unsigned length)
/* This is used in the tapering mode. */
/*************************************/
{
	if(length >= 100) return 1.0e-06;
	else if(length >= 70) return 1.0e-05;
	else if(length >= 40) return 1.0e-04;
	else return 1.0e-03;
}
/*************************************/


/*************************************/
int thr_level(double thr1)
/* This function returns thr level as defined 
 * by the thr_list. */
/*************************************/
{
	int i;

	for(i=0;i<=num_thr;i++){
		if(thr1 <= thr_list[i]) break;
	}
	return i;
}
/*************************************/

		
/**********************************************************************/
unsigned Evaluation1(unsigned x)

/* 351
 * This is used to evaluate abundance in AssignSingletons (node.c)
 * 
 * The function is explicitly written as:
 *
 * y = (x + 4) / 3
 * x 1 2 3 4 5 6 7 8 9
 * y 1 2 2 2 3 3 3 4 4 
 */
/**********************************************************************/
{
	unsigned y;

	if(x == 0) y = 0;		
	else if(x == 1) y = 1;
	else if(x == 2) y = 2;
	else{
		y = generalEvaluationFunction(x, 4.0/3.0, 1.0/3.0, 0.0, 0.0, 0.0);
	}

	return y;	
}		
/**********************************************************************/


/**********************************************************************/
unsigned Evaluation2(unsigned x)

/* 351
 * This is used to evaluate abundance in AssignSingletons (node.c)
 * 
 * The function is explicitly written as:
 *
 * y = 1 + (x + log x) / 2
 * x 1 2 3 4 5 6 7 8 9
 * y 1 2 2 3 3 4 4 5 5 
 */
/**********************************************************************/
{
	unsigned y;

	if(x == 0) y = 0;		
	else if(x == 1) y = 1;
	else if(x == 2) y = 2;
	else{
		y = generalEvaluationFunction(x, 1.0, 0.5, 0.5, 0.0, 0.0);
	}

	return y;	
}		
/**********************************************************************/



/**********************************************************************************/
unsigned generalEvaluationFunction(unsigned x,double a, double b,double c,double d,double e)

/* 351
 * This is a composite function as defined by the following equation:
 *
 * y = a + b*x + c*log(x) + d*x1^e.
 *
 */
/**********************************************************************************/
{
	double x1 = (double)x;
	double y1;

	if(e > 1.0) e = 1.0;
	else if(e < 0.01) e = 0.0;
	if(b > 1.0) b = 1.0;
	else if(b < -1.0) b = -1.0;

	y1 = a + b * x1 + c * log(x1) + d * pow(x1,e);

	if(y1 > x1) y1 = x1;
	else if(y1 < 0.5) y1 = 0.0;

	return (unsigned)y1;
}
/**********************************************************************************/

int only1_flag;
int keep_data;
/**********************************************************************************/
int select_initial_clique(int m0, int n0, int delta_max)

/* 351 */
/* This subroutine is used to select members of a clique formed around
 * a local maximum. */
/**********************************************************************************/
{
	int x0,y0,d,dx,dy,xm,ym;
	int counter,total_count;
	int x,y;
	int m,n,mm,nn,m1,n1,min_z,opt_m1,opt_n1;
	int flag;
	int allow_reversal = 2;
	int x1,x2,y1,y2,y2a;
	int s;
	int xysize=11*(num_thr+1);
	static int repeat;
	static int bridging;
	int max_radius = 4;
	int max_radius2 = max_radius * max_radius;


	if(m0 < _min_overlap || m0 > 10) {
		fprintf(stderr,"m0 is strange: m0=%d.\n",m0);
		m0 = _min_overlap;
	}
	if(n0 < 0 || n0 > num_thr) {
		fprintf(stderr,"n0 is strange: n0=%d.\n",n0);
		n0 = num_thr / 2;
	}
	
	d = 1;
	x0 = m0 - d;
	y0 = n0 - d;
	dx = 1;
	dy = 1;
	xm = m0 + d;
	ym = n0 + d;
	counter = 0;

	

	if(keep_data == 0){	
		repeat = 0;
		bridging = 0;
		for(m=0;m<11;m++){
			for(n=0;n<=num_thr;n++){
				z[n*11+m] = -1;
			}
		}
	}else{
		repeat += 1;
	}

	if(repeat >= 2) return (n0*11+m0);


	z[n0*11+m0] = 0;
	for(x=m0,y=y0;d<12;x+=dx,y+=dy){

/*
		printf("x=%d, y=%d   ",x,y);
*/

		if(x == xm && y == n0){
			dx = -1;
			dy = 1;
		} else if(y == ym && x == m0){
			dx = -1;
			dy = -1;
		} else if(x == x0 && y == n0){
			dx = 0;
			dy = -1;
		} else if(x == x0 && y == n0 - 1){
			dx = 1;
			dy = -1;
		} else if(x == m0 && y == y0 - 1){
			dx = 1;
			dy = 1;
			if(counter == 0) {
				if(flag == 1) break;
				else {
					flag = 1;
				}
			} else flag = 0;
					
			d++;
			if(d > 11 && d > num_thr) break;
			x0--;
			y0--;
			counter = 0;
			xm++;
			ym++;
		}

		/* overlap score is limited above 0.3 */
		if(x < _min_overlap-1 || x > 10 || y < 0 || y >= num_thr) continue;

		/* Bridging 0 352a4 */
		if(xy[y*11+x] == 0){
			if(bridging == 0 && dx == -1){
				if(dy == -1){	/* moving up left */
					if(x < 10 && xy[y*11+x-1] > 0 && z[y*11+x+1] > -1){		/* left */
						z[y*11+x] = z[y*11+x+1] + 1;
						bridging = 1;
					}else if(y > 0 && y < num_thr && xy[(y+1)*11+x] > 0 && z[(y-1)*11+x] > -1){		/* down */
						z[y*11+x] = z[(y-1)*11+x] + 1;
						bridging = 1;
					}else if(x < 10 && y > 0 && y < num_thr && xy[(y+1)*11+x-1] > 0 && z[(y-1)*11+x+1] > -1){	/* down left */
						z[y*11+x] = z[(y-1)*11+x+1] + 1;
						bridging = 1;
					}
				} else if(dy == 1){	/* moving down left */
					if(y > 0 && y < num_thr && xy[(y+1)*11+x] > 0 && z[(y-1)*11+x] > -1){		/* down */
						z[y*11+x] = z[(y-1)*11+x] + 1;
						bridging = 1;
					}
				}
			}
			continue;
		}
		min_z = -1;
		for(m1=-1;m1<=1;m1++){
			for(n1=-1;n1<=1;n1++){
				if(m1==0 && n1==0) continue;
				if((mm=x+m1) < 2 || mm > 10) continue;
				if((nn=y+n1) < 0 || nn >= num_thr) continue;
				if(xy[nn*11+mm] == 0 && z[nn*11+mm] == -1) continue;
				if(xy[nn*11+mm] + 1 < xy[y*11+x] &&  xy[nn*11+mm] > 0 && (xy[(y+1)*11+x] != 0 || xy[y*11 + x -1] != 0)) continue;
				else if(xy[nn*11+mm] + 1 == xy[y*11+x]){
					if(xy[nn*11+mm] > 0){
						if(allow_reversal <= 0) continue;
						else allow_reversal--;
					}
				}
				if(z[nn*11+mm] > -1){ 
					if(min_z == -1 || z[nn*11+mm] < min_z){
						min_z = z[nn*11+mm];
						opt_m1 = m1;
						opt_n1 = n1;
						continue;
					}
				}
			}
		}
		if(min_z > -1){
			if(opt_n1 == 0 || opt_m1 == 0) z[y*11+x] = min_z + 2;
			else z[y*11+x] = min_z + 3;
			counter++;
		} 

	}

	/* 352f7 maximum reach is limited within max_radius. */
	for(x=0;x<=10;x++){
		for(y=num_thr;y>=0;y--){
			if(y <= n0 && x >= m0) continue;
			if(z[y*11+x] > -1){

				/* 352f12 */
				if(10 + y - x > delta_max){
					z[y*11+x] = -1;
					continue;
				}
				if((m0 - x)*(m0 - x) + (y - n0)*(y - n0) > max_radius2) z[y*11+x] = -1;
			} 
		}
	}
	/* This also removes '1' generated by bridging. 352f13 */
	for(x=0;x<=10;x++){
		for(y=num_thr;y>=0;y--){
			if(z[y*11+x] > -1 && xy[y*11+x] == 0) z[y*11+x] = -1;
		}
	}


	/* Search for lower edge */

	for(x=0;x<11;x++){
		for(y=num_thr;y>=0;y--){
			if(z[y*11+x] > -1) break;
		}
		if(y >= 0) break;
	}
	x1 = x;
	y1 = y;
	for(y=num_thr;y>=0;y--){
		for(x=0;x<11;x++){
			if(z[y*11+x] > -1) break;
		}
		if(x < 11) break;
	}
	x2 = x;
	y2 = y;

	if(x2 < x1) x2 = x1;
	if(y2 < y1) y2 = y1;

/*
printf("\nx1=%d,y1=%d,x2=%d,y2=%d.\n",x1,y1,x2,y2);
*/
	/* Setting non -1 value for all points located upper and right */

	for(x=x1;x<11;x++){
		if(x < x2){
/*			y2a = ((double)(y2 - y1)/(double)(x2 - x1)) * (double)(x - x1) + x1;
debug */
			y2a = ((double)(y2 - y1)/(double)(x2 - x1)) * (double)(x - x1) + y1;
			for(y=y2a;y>=0;y--){
				if(z[y*11+x] == -1 && xy[y*11+x] > 0){
					z[y*11+x] = 1;
				}
			}
		} else {
			for(y=y2;y>=0;y--){
				if(z[y*11+x] == -1 && xy[y*11+x] > 0){
					z[y*11+x] = 1;
				}
			}
		}
	}

	/* Counting number of species */
	if(use_org){
		counter = 0;
		for(s=1;s<=num_org_all;s++){
			for(m=10;m>=0;m--){
				for(n=0;n<=num_thr;n++){
					if(z[n*11+m] > -1){
						if(sp[(xysize * s) + (n*11+m)] > 0){
							counter += 1;
							if(print_2D_tables) {
								printf("%s ",OrgPrefix(s));
								fflush(stdout);
							}
							break;
						}
					}
				}
				if(n<=num_thr) break;
			}
		}

		total_count = CountOrg(_min_overlap,num_thr,1);


/*	For debug 
		for(m=10;m>=0;m--){
			for(n=0;n<=num_thr;n++){
				for(s=1;s<=num_org_all;s++){
					sp[n*11+m] += sp[(xysize * s) + (n*11+m)];
				}
			}
		}	
		printf("\n");
		printf("sp-Table\n");	
		printf("thr     ");
		for(m=0;m<11;m++) printf("%3.1f ",0.1*m);
		printf("\n");
	
		for(n=0;n<=num_thr;n++){
			printf("%6.1e",thr_list[n]);
			for(m=0;m<11;m++){
				printf("%3d ",sp[n*11+m]);
			}
			printf("\n");
		}
		printf("\n");
		fflush(stdout);
*/
	}
	
	/* Printing table */

	if(print_2D_tables){

		printf("\n");
		printf("z-Table\n");	
		printf("thr     ");
		for(m=0;m<11;m++) printf("%3.1f ",0.1*m);
		printf("\n");

		for(n=0;n<=num_thr;n++){
			printf("%6.1e",thr_list[n]);
			for(m=0;m<11;m++){
				printf("%3d ",z[n*11+m]);
			}
			printf("\n");
		}
		if(use_org){
			printf("Included species: %d per total: %d ",counter,total_count);
		}
		printf("\n");
		fflush(stdout);

	}
/* 352f13	
	if(use_org){
		if(total_count > 2 && counter * 1.5 < total_count){
			flag = 0;
			for(offset=x1-y2;offset>x1-y2-5 || offset>=0;offset--){
				for(m=10;m>=_min_overlap;m--){
					n = m - offset;
					if(n < 0 || n >= num_thr) continue;
					if(z[n*11+m] == -1 && xy[n*11+m] > 0){
						flag = 1;
						break;
					}
				}
				if(flag == 1) break;
			}
			if(flag == 1){
				keep_data = 1;
				result = select_initial_clique(m,n,delta_max);
				x1  = result % 11;
				y2  = result / 11;
			}
		}
	}
*/
	return (y2 * 11 + x1);

}
/**********************************************************************************/

/**********************************************************************************/
int DeltaTest(Node **a, unsigned nodes, unsigned i, unsigned j)
/**********************************************************************************/
{
	int delta;
	int k1,k2,k3,x1,x2,x3;
	int k,m,n;
	int max_delta = 11 + num_thr - _min_overlap;
	int delta_dist_ii[max_delta+1];
	int delta_dist_jj[max_delta+1];
	int delta_dist_ij[max_delta+1];
	int items_ii = 0;
	int items_jj = 0;
	int items_ij = 0;
	double total_i,total_j;
	int subtotal;
	double score;
	double overlap;
	unsigned n7;
	unsigned n4i=a[i]->n4;
	unsigned n4j=a[j]->n4;
	double min_delta_overlap = _min_delta_overlap;
	int delta_offset = max_delta / 10;

/*	double min_delta_overlap = 0.2;
*/

	for(k1=0;k1<=max_delta;k1++){
		delta_dist_ii[k1] = 0;
		delta_dist_jj[k1] = 0;
		delta_dist_ij[k1] = 0;
	}

	for(k1=0;k1<n4i;k1++){
		x1 = a[i]->idlist4[k1];
		x2 = FindNode(a,nodes,x1);
		if((delta = a[x2]->delta_max) >= 0 && delta <= max_delta){
			delta_dist_ii[delta] += 1;
			items_ii += 1;
		}
	}
	for(k1=0;k1<n4j;k1++){
		x1 = a[j]->idlist4[k1];
		x2 = FindNode(a,nodes,x1);
		if((delta = a[x2]->delta_max) >= 0 && delta <= max_delta){
			delta_dist_jj[delta] += 1;
			items_jj += 1;
		}
	}

	
	for(k1=0;k1<n4i;k1++){
		x1 = a[i]->idlist4[k1];
		x2 = FindNode(a,nodes,x1);
		for(k2=0;k2<n4j;k2++){
			x3 = a[j]->idlist4[k2];
			n7 = a[x2]->n7;
			for(k3=0;k3<n7;k3++){
				if(a[x2]->matchlist7[k3].ID == x3){
					score = a[x2]->matchlist7[k3].score;
					overlap = a[x2]->matchlist7[k3].overlap;
					if(overlap >= 1.0) m = 10;
					else if(overlap >= 0.9) m = 9;
					else if(overlap >= 0.8) m = 8;
					else if(overlap >= 0.7) m = 7;
					else if(overlap >= 0.6) m = 6;
					else if(overlap >= 0.5) m = 5;
					else if(overlap >= 0.4) m = 4;
					else if(overlap >= 0.3) m = 3;
					else if(overlap >= 0.2) m = 2;
					else if(overlap >= 0.1) m = 1;
					else if(overlap >= 0.0) m = 0;
					else m = 0;
					
					for(k=0;k<=num_thr;k++){
						if(score <= thr_list[k]){
							  n	= k;
							  break;
						}
					}
					if(k>=num_thr) n = num_thr;
					delta = n + 10 - m;
					break;
				}
			}
			if(k3 == n7){
				delta = max_delta;
			}
			delta_dist_ij[delta] += 1;
			items_ij += 1;
		}
	}

	total_i = 0.0;
	for(delta=0;delta<=max_delta;delta++){
		if(delta_dist_ii[delta]==0) continue;
		subtotal = 0;
/*		for(k1=0;k1<=delta;k1++){
*/
		for(k1=0; k1 <= (delta + delta_offset) && k1 <= max_delta; k1++){
			subtotal += delta_dist_ij[k1];
		}
		total_i += (double)subtotal * (double)delta_dist_ii[delta] / (double)(items_ii * items_ij);
	}

	total_j = 0.0;
	for(delta=0;delta<=max_delta;delta++){
		if(delta_dist_jj[delta]==0) continue;
		subtotal = 0;
/*		for(k1=0;k1<=delta;k1++){
*/
		for(k1=0; k1 <= (delta + delta_offset) && k1 <= max_delta; k1++){
			subtotal += delta_dist_ij[k1];
		}
		total_j += (double)subtotal * (double)delta_dist_jj[delta] / (double)(items_jj * items_ij);
	}

#if 0 
	printf("Check for delta  i:%s j:%s\n",a[i]->name,a[j]->name);
	printf("\ndelta ii   jj   ij\n");
	for(delta=0;delta<=max_delta;delta++){
		printf("%d %5d %5d %5d\n",delta,delta_dist_ii[delta],delta_dist_jj[delta],delta_dist_ij[delta]);
	}
	printf("items_ii=%d, items_jj=%d\n\n",items_ii,items_jj);
	printf("\n");
	printf("total_i=%f, total_j=%f\n\n",total_i,total_j);
#endif

	if(total_i > min_delta_overlap && total_j > min_delta_overlap) return TRUE;
	else if(total_i * total_j > min_delta_overlap * min_delta_overlap * 2.0) return TRUE;
	
	else return FALSE;
}
/**********************************************************************************/

/**********************************************************************************/
int CountOrgInIDlist(Node **a,unsigned nodes,unsigned i,unsigned j)
/* This routine counts number of organisms that are included in the IDlist4 
 * of nodes i and j, and estimates increment in  number of organisms upon
 * fusion of nodes i and j. If j = 0, then number of organisms that are not included
 * in a[i]->idlist4 is counted.
 * Return value is 1, if two nodes should be merged, while it is 0 if two nodes
 * should not be merged. Negative return value indicates error.*/
/* Regroup level (version 3.5.3f) 1 (weak) - 10 (hard). 5 is standard.*/
/* 355: To avoid too many proteins of a limited number of species,
 *      entropic measure is used. Older codes were inactivated. Aug. 2007 */
/**********************************************************************************/
{
	int s,m;
	unsigned n,k,offset;
	unsigned n4i, n4j;
	int xysize = (num_thr + 1) * 11;
	int count0,count1,count2,count0and1,count0xor1;
	double org_measure = 3.0;
	double allowance_level = 0.8;
	double inf0,inf1,inf2,JSdist,JSmax,JSmax0;
	double total0,total1,total2;
	int return_value = 0;
	double JS_level = _JS_level_0;
	Boolean small_addition = FALSE;
	unsigned SA = num_org/10 + 1;

/*	353h standard values:
	double _org_measure_0 = 5.0;
	double _org_measure_1 = 0.3;
	double _allowance_level_0 = 0.65;
	double _allowance_level_1 = 0.025;
	double _JS_level_0 = 0.07;
	double _JS_level_1 = 0.01;
*/

	if(SA < 3) SA = 3;
	/* 353f */	
	if(_regroup_level >= 0){
		if(_regroup_level > 12) _regroup_level = 12;
		JS_level = _JS_level_0 + _JS_level_1 * (double)_regroup_level;
		org_measure = _org_measure_0 - _org_measure_1 * (double)_regroup_level;
		allowance_level = _allowance_level_0 + _allowance_level_1 * (double)_regroup_level;
	}

	count0 = count1 = count2 = 0;

	if(!use_org) return -1;	
	if(i > nodes || j > nodes || i == j) return -1;
	if((n4i = a[i]->n4) == 0 || a[i]->n3 == 0) return -1;
	if(j > 0 && ((n4j = a[j]->n4) == 0 || a[j]->n3 == 0)) return -1;
	if((n4i >= num_org/2 && n4j < SA) || (n4i < SA && n4j >= num_org/2)) small_addition = TRUE;
	else small_addition = FALSE;

/*	for(m=0;m < xysize * (num_org_all + 1);m++) sp[m] = 0;
 *	*/
	for(s=0;s<num_org_all;s++){
		sp[(xysize * s)] = sp[(xysize * s) + 1] = sp[(xysize * s) + 2] = 0;
	}

	/* positions 0, 1 and 2 are used for node i, node j and combined node i and j. */

	offset = 0;
	for(k=0;k<n4i;k++){
		n = FindNode(a,nodes,a[i]->idlist4[k]);
		sp[(xysize * (a[n]->species - 1)) + offset] += 1;
	}

	offset = 1;
	for(k=0;k<n4j;k++){
		n = FindNode(a,nodes,a[j]->idlist4[k]);
		sp[(xysize * (a[n]->species - 1)) + offset] += 1;
	}

	for(s=0;s<num_org_all;s++){
		sp[(xysize * s) + 2] = sp[(xysize * s)] + sp[(xysize * s) + 1];
	}


/********Not used now***********************
	for(s=0;s<num_org_all;s++){
		if(sp[(xysize * s)] > 0) count0 += 1;
		if(sp[(xysize * s) + 1] > 0) count1 += 1;
		if(sp[(xysize * s) + 2] > 0) count2 += 1;
		if(sp[(xysize * s)] > 0 && sp[(xysize * s) + 1] > 0) count0and1 += 1;
		if((sp[(xysize * s)] > 0 && sp[(xysize * s) + 1] == 0) || (sp[(xysize * s)] == 0 && sp[(xysize * s) + 1] > 0)) count0xor1 += 1;
	}
********************************************/

	/* 355 */
	inf0 = inf1 = inf2 = total0 = total1 = total2 = 0.0;
	for(s=0;s<num_org_all;s++){
		if(sp[(xysize * s) + 2] > 0){
			total2 += (double)sp[(xysize * s) + 2];
			inf2 += (double)sp[(xysize * s) + 2] * log((double)sp[(xysize * s) + 2]);
			total0 += (double)sp[(xysize * s)];
			if(sp[(xysize * s)] == 0){
				inf0 += 0.0;
			} else inf0 += (double)sp[(xysize * s)] * log((double)sp[(xysize * s)]);
			total1 += (double)sp[(xysize * s) + 1];
			if(sp[(xysize * s) + 1] == 0){
				inf1 += 0.0;
			} else inf1 += (double)sp[(xysize * s) + 1] * log((double)sp[(xysize * s) + 1]);
		}
	}
	/* This is information content, which is the difference from the maximal value */
	inf0 = log(total0) - inf0 / total0;
	inf1 = log(total1) - inf1 / total1;
	inf2 = log(total2) - inf2 / total2;

	JSdist = inf2 - (inf0 * total0)/total2 - (inf1 * total1)/total2;
	JSmax = log(total0 + total1) - (total0 * log(total0))/(total0 + total1) - (total1 * log(total1))/(total0 + total1);

	/* Addition of (num_org-2) and 2 species are allowed. */

	JSmax0 = log((double)num_org) - (double)(num_org - 2) * log((double)(num_org - 2))/(double)num_org - 2.0 * log(2.0)/(double)num_org;

	if(!small_addition){
		if((total0 >= num_org/2 && total1 < SA) || (total0 < SA && total1 >= num_org/2)) small_addition = TRUE;
	}

	if(small_addition){
	   if(JSdist < JSmax0) return 0;
	} else {
		if(JSdist < 0.5 * log(2.0)) return 0;
	}
	printf("JSdist=%f JSmax=%f JSmax0=%f inf0=%f, %s n4=%u, inf1=%f, %s n4=%u\n",\
			JSdist,JSmax,JSmax0,inf0,a[i]->name,a[i]->n4,inf1,a[j]->name,a[j]->n4);
	if(JSmax - JSdist < JS_level * JSmax) return 3;
	else return 2;

#if 0
	/* The following lines are not executed. */

	/* num_org=number_of_genomes but not num_org_all in this context */
	/* no overlap in species */
	if(count0and1 <= 1 && count0 > 1 && count1 > 0) return_value = 2;
	/* increase should be large enough, more than a fraction of org_measure */
	else if((double)(count2 - count0) < (double)(num_org - count0)*org_measure/(double)num_org) return_value = 1;
	/* In the next two if-phrases, OR was changed to AND. 353g */
	/* too much overlap in species */
	else if((double)count0and1 > allowance_level * (double)count0 && (double)count0and1 > allowance_level * (double)count1) return_value = 1; 
	else if((double)count0xor1 < (double)(num_org - count0)*org_measure/(double)num_org && (double)count0xor1 < (double)(num_org - count1)*org_measure/(double)num_org) return_value = 1;
	else return_value = 2;

	return return_value;
#endif
}
/**********************************************************************************/

/**********************************************************************************/
int CountOrg(int m1,int n1,int mode)
/* This function evalutates number of species that are included in the region of 
 * xy as defined by the mode. 
 * Mode 1, a rectangular area having the point (m1,n1) at its down left corner.
 * Mode 2, a triangular area above the 45-degree line through the point (m1,n1). 
 */
/**********************************************************************************/
{
	int count = 0;
	int s,m,n;
	int xysize = (num_thr + 1) * 11;
	int n_max,n_limit;

	if(m1 > 10 || m1 < 0 || n1 > num_thr || n1 < 0){
		fprintf(stderr,"Strange value of m1 and/or n1 in CountOrg.\n");
		return 0;
	}

	if(mode == 1){
		for(s=1;s<=num_org_all;s++){
			for(m=10;m>=m1;m--){
				for(n=0;n<=n1;n++){
					if(sp[(xysize * s) + (n*11+m)] > 0){
						count += 1;
						break;
					}
				}
				if(n<=n1) break;
			}
		}
	} else if(mode == 2){
		n_max = n1 + 10 - m1;
		for(s=1;s<=num_org_all;s++){
			n_limit = n_max;
			for(m=10;m>=0;m--,n_limit--){
				for(n=0;n<=n_limit;n++){
					if(n >= num_thr || n < 0) continue;
					if(sp[(xysize * s) + (n*11+m)] > 0){
						count += 1;
						break;
					}
				}
				if(n<=n_limit) break;
			}
		}

	} else {
		fprintf(stderr,"Strange value of mode in CountOrg.\n");
		return 0;
	}
				
	return count;
}
/**********************************************************************************/


/**********************************************************************************/
int OptimizeOrg(void)
/* This function returns the point in the xy-table, above which the number of
	organisms is optimal. Return value -1 is used to indicate anomaly.*/
/* version 355w: new evaluation function is used. */
/**********************************************************************************/
{
	int m,n,offset,offset2;
	int total_count,counter,temp;
	int levelA2;
	double levelA;
	double f1 = _f1;
	double f2 = _f2;
	double f3 = _f3;
	int f4 = (int)(1.0 / f3) - 1;
	double f5;
	double tn;
	int minimum_y = 0;

	total_count = CountOrg(0,num_thr,1);
	if(print_2D_tables){
		printf("Number of organisms in OptimizeOrg: total_count=%d, num_org=%d,",total_count,num_org);
	}
	if(total_count <= 2){
		if(print_2D_tables) printf("\n");
		return (-1);
	}
	tn = (double)total_count/(double)num_org;


	if(f1 == 0.0 && f2 == 0.0) goto NEW;

/* This part is the old version */
	if(f2 - f1 <= 0.0 || 1.0 - f3 <= 0.0){
		fprintf(stderr,"Strange values for f1, f2 or f3 in OptimizeOrg.\n");
		return (-1);
	}

	/* num_org=number_of_genomes but not num_org_all in this context */
	if((double)total_count < (double)num_org * f1){
		levelA = (double)(total_count + f4) * f3;
	} else if((double)total_count < (double)num_org * f2){
		f5 = ( (1.0 - f3) * (tn - f1) / (f2 - f1) ) + f3;
		levelA = (double)total_count * f5;
	} else if((double)total_count >= (double)num_org * f2){
		levelA = (double)total_count;
	} else {
		if(print_2D_tables) printf("\n");
		return -2;
	}
	levelA2 = (int)levelA;
	if(levelA2 > num_org) levelA2 = num_org;

	goto COMMON;

NEW:
	if(print_2D_tables) printf("<new>");
	if(f3 <= 0.0){
		fprintf(stderr,"Strange value for f3 in OptimizeOrg.\n");
		return (-1);
	}

	/* For small proteins, a smaller value for f3 is used. 355x */
	minimum_y = getMinimumY();
	if(minimum_y > 5 || minimum_y > (num_thr / 2)){
		f3 = f3 * 0.6;
	}
	
	f1 = 2.0 / (double)num_org;
	f5 = f1 + (1.0 - f1) * exp(f3 * log(tn));
	/* rounding */
	levelA = (double)total_count * f5 + 0.9999;
	levelA2 = (int)levelA;
	if(levelA2 > num_org) levelA2 = num_org;
	f3 = _f3;

COMMON:	
	counter = 0;
	for(offset=0;offset<num_thr+11;offset++){
		offset2 = (offset + 1)/2;
		m = 10 - offset2;
		n = offset / 2;
		/* 352f10 */
		if(m < _min_overlap && n > num_thr) break;
		if(m < _min_overlap) m = _min_overlap;
		if(n > num_thr) n = num_thr;
		if(counter <  (temp=CountOrg(m,n,2))){
			counter = temp;
			if(counter >= levelA2){
				break;
			}
		}
	}
/*	if(counter == 0){	*/
	if(counter < (int)levelA){
		if(print_2D_tables) printf(", Low counter");
/*		return -3;
*/
	} else if(offset >= num_thr+11 || m < 0 || n > num_thr){
		if(print_2D_tables) printf("\n");
		return -4;
	}

	if(print_2D_tables) printf(", levelA=%d, counter=%d\n",(int)levelA,counter);

	return (n*11+m);	
}
/**********************************************************************************/


/**********************************************************************************/
int getMinimumY(void)
/* This returns the y value for the best point in 2D table. */
/**********************************************************************************/
{
	int minimum_y;
	int x,y;
	int flag = 0;

	if(xy[10] != 0) minimum_y = 0;
	else {
		flag = 0;
		for(y=0;y<=num_thr;y++){
			for(x=10;x>=_min_overlap;x--){
				if(xy[y*11+x] != 0){
					minimum_y = y;
					flag = 1;
					break;
				}
			}
			if(flag) break;
		}	
	}

	return minimum_y;
}
/**********************************************************************************/

/**********************************************************************************/
int makeInfContTable(int mode)
/* mode 1: triangular region.
 * mode 2: rectangular region.
 * mode 3: circular region.
 355: For small proteins, upper areas filled with 0 are not counted.
 Returns minimum_y, indicating the upper right edge of the xy table. */
/**********************************************************************************/
{
	int x,y;
	int delta;
	int flag = 0;
	int count,total;
	double pi,temp;
	int max_delta = 11 + num_thr - _min_overlap;
	int minimum_y = 0;	/* 355 */

	/* 355 skip upper areas filled with 0 */
	/*
	if(xy[10] != 0) minimum_y = 0;
	else {
		flag = 0;
		for(y=0;y<=num_thr;y++){
			for(x=10;x>=_min_overlap;x--){
				if(xy[y*11+x] != 0){
					minimum_y = y;
					flag = 1;
					break;
				}
			}
			if(flag) break;
		}	
	}	 355 */
	minimum_y = getMinimumY();

/*	if(minimum_y != 0) printf("minimum_y=%d \n",minimum_y);
*/

	for(delta=1;delta<=max_delta;delta++){
		total = 0;
		count = 0;
		for(x=10;x>=_min_overlap;x--){
/*			for(y=0;y<=num_thr;y++){
			for modes 2 and 3, y is now replaced by y - minimum_y*/
			for(y=minimum_y;y<=num_thr;y++){
				flag = 0;
				if(mode == 1 && y <= x + delta - 10) flag = 1;
				else if(mode == 2 && y - minimum_y <= delta && x >= 10-delta) flag = 1;
				else if(mode == 3 && ((10-x)*(10-x) + (y - minimum_y)*(y - minimum_y) <= delta*delta) ) flag = 1;
/*				else if(mode == 2 && y <= delta && x >= 10-delta) flag = 1;
				else if(mode == 3 && ((10-x)*(10-x) + y*y <= delta*delta) ) flag = 1;
*/
				if(flag == 1){
					count += 1;
					total += xy[y*11+x];
				}
			}
		}

		if(count <= 0) inf[delta][0] = 0.0;
		else inf[delta][0] = log10((double)count);
		/* This does not occur now. 355 */	
		if(total == 0) {
			if(print_2D_tables){
				printf("total=0 for delta=%d in makeInfContTable.\n",delta);
			}
			inf[delta][1] = inf[delta][3] = inf[delta][4] = 0.0;
			inf[delta][2] = inf[delta][0];
			continue;
		}
		inf[delta][4] = (double)total;

		temp = 0.0;
		for(x=10;x>=_min_overlap;x--){
/*			for(y=0;y<=num_thr;y++){
			for modes 2 and 3, y is now replaced by y - minimum_y*/
			for(y=minimum_y;y<=num_thr;y++){
				flag = 0;
				if(mode == 1 && y <= x + delta - 10) flag = 1;
				else if(mode == 2 && y - minimum_y <= delta && x >= 10-delta) flag = 1;
				else if(mode == 3 && ((10-x)*(10-x) + (y - minimum_y)*(y - minimum_y) <= delta*delta) ) flag = 1;
				if(flag == 1){
					if(xy[y*11+x] > 0){
						pi = (double)xy[y*11+x] / (double)total;
						temp += (-1.0) * pi * log10(pi);
					}
				}
			}
		}
		inf[delta][1] = temp;
		inf[delta][2] = inf[delta][0] - inf[delta][1];
	
		if(use_org){
			if(delta <= num_thr){
				inf[delta][3] = CountOrg(10,delta,2);
			} else {
				inf[delta][3] = CountOrg((10 - (delta-num_thr)),num_thr,2);
			}
		} else inf[delta][3] = 0;
	}

/*	if(print_2D_tables || (minimum_y != 0)){
*/
		if(print_2D_tables){
		printf("\n");
		printf("delta   Smax      S    Smax-S org count\n");
		for(delta=1;delta<max_delta;delta++){
			printf("%5d  %6.3f  %6.3f  %6.3f  %3.0f  %3.0f\n",delta,inf[delta][0],inf[delta][1], \
					inf[delta][2],inf[delta][3],inf[delta][4]);
		}
		printf("\n");
	}

	return minimum_y;
	
}
/**********************************************************************************/


/**********************************************************************************/
int selectDeltaFromInf(int delta7,int minimum_y)
/* For small proteins, scanning of y starts from the minimum_y, the upper right
 * corner. 355r */
/**********************************************************************************/
{
	int delta;
	double S1,S_max,dS1,dS_max,dS_min;

	int delta0;	/* Initial rise in S: initial maximal increase */
	int delta1;	/* Peak in dS, Smax - S, after delta0 */
	int delta2;	/* Valley in dS after delta1 */
	int delta3;	/* Point at which count exceeds number_of_genomes */
	int delta4;	/* Initial maximal increse in count */
	int delta5;	/* Initial non-zero value of count */
	int delta6;	/* Point at which S reaches the plateau */
/*	int delta7;	(org only) the level determined by OptimizeOrg */

	int max_delta = 11 + num_thr - _min_overlap;
	int flag1;
	double zero = 0.005;
	double final_dS = inf[max_delta-1][2];
	int d_count = 0;
	int initial_d_count_max = inf[max_delta-1][4] / 5;
	int d_count_max = initial_d_count_max;

	int delta_min = minimum_y;	/* 355r */
	int delta_max = minimum_y;	/* 355r */
	/* Both min and max are to be returned as 100*min+max */

	delta0 = delta1 = delta2 = delta3 = delta4 = delta5 = delta6 = 0;
	if(inf[max_delta-1][4] < number_of_genomes) delta3 = max_delta;
	S_max = 0.0;
	dS_max = 0.0;
	dS_min = 0.0;
	flag1 = 0;

	for(delta=minimum_y+1;delta<max_delta;delta++){
/*	for(delta=1;delta<max_delta;delta++){ */
		S1 = inf[delta][1] - inf[delta-1][1];
		dS1 = inf[delta][2] - inf[delta-1][2];

		if(delta3 == 0){
			if(inf[delta][4] > number_of_genomes) delta3 = delta;
		}

		if(delta5 == 0){
			if(inf[delta][4] > zero) delta5 = delta;
		}

		d_count = inf[delta][4] - inf[delta-1][4];

		if(flag1 >= 1 && delta4 == 0){
			if(d_count > d_count_max){
				d_count_max = d_count;
			} else if(d_count_max > initial_d_count_max && d_count <= initial_d_count_max){ 
				delta4 = delta - 1;
			}
		}
		if(flag1 == 6) continue;
		if(flag1 == 0){
			if(S1 > S_max) {
				S_max = S1;
				flag1 = 1;
				delta--;
			}
		} else if(flag1 == 1){
			if(S1 > S_max) {
				S_max = S1;
			} else if(S1 < S_max){
				flag1 = 2;
				delta--;
				delta0 = delta;
			}
		} else if(flag1 == 2){
			if(dS1 > 0.0){
				flag1 = 3;
			}
		} else if(flag1 == 3){
			if(dS1 > 0.0){
				continue;
			} else {
				delta1 = delta - 1;
				flag1 = 4;
			}
		} else if(flag1 == 4){
			if(dS1 + zero < 0.0){
				if(inf[delta][2] >= final_dS - zero && inf[delta][2] <= final_dS + zero){
					flag1 = 5;
					delta2 = delta;
					flag1 = 6;
					continue;
				}
			} else {
				flag1 = 5;
				delta2 = delta - 1;
				flag1 = 6;
				continue;
			}
		}
	}
	/* 355r */
	if(flag1 == 3) delta1 = delta2 = max_delta - 1;

	flag1 = 0;
	S1 = inf[max_delta-1][1];
	for(delta=max_delta-1;delta>0;delta--){
		if(flag1 >= 5){
			if(inf[delta-1][1] < S1){
				delta6 = delta;
				break;
			}
		} else if(inf[delta-1][1] == S1){
			flag1 += 1;
		} else {
			flag1 = 0;
		}
	}

	if(print_2D_tables){
		printf("minimum_y= %d, delta0 = %d, delta1 = %d, delta2 = %d, delta3 = %d, delta4 = %d, delta5 = %d, delta6 = %d, delta7 = %d.\n",\
			minimum_y,delta0,delta1,delta2,delta3,delta4,delta5,delta6,delta7);
	}

	if(inf[max_delta-1][1] < zero){
		delta_min = delta5;
		delta_max = delta5;
		/* 355r added minimum_y for small proteins */
	} else if(inf[(minimum_y + num_thr + 10 - _min_overlap)/2][1] < zero){
		delta_min = delta5;
		delta_max = delta5;
	} else if(inf[max_delta-1][1] < log10(2.0) + zero){
		delta_min = delta0;
		delta_max = delta0;
	} else if(delta1 == 0 || delta2 == 0){

#if 0
		if(inf[max_delta-1][1] < zero){
			delta_min = delta5;
			delta_max = delta5;
		/* 355r added minimum_y for small proteins */
		} else if(inf[(minimum_y + num_thr + 10 - _min_overlap)/2][1] < zero){
			delta_min = delta5;
			delta_max = delta5;
		} else if(inf[max_delta-1][1] < log10(2.0) + zero){
			delta_min = delta0;
			delta_max = delta0;
		} else {
#endif

			delta_min = max_delta;
			if(delta0 > 0 && delta0 < delta_min) delta_min = delta0;
			if(delta3 > 0 && delta3 < delta_min) delta_min = delta3;
			if(delta4 > 0 && delta4 < delta_min) delta_min = delta4;
			if(delta6 > 0 && delta6 < delta_min) delta_min = delta6;
			if(delta7 > 0 && delta7 < delta_min) delta_min = delta7;

			delta_max = max_delta;
			if(delta0 > delta_min && delta0 < delta_max) delta_max = delta0;
			if(delta3 > delta_min && delta3 < delta_max) delta_max = delta3;
			if(delta4 > delta_min && delta4 < delta_max) delta_max = delta4;
			if(delta6 > delta_min && delta6 < delta_max) delta_max = delta6;
			if(delta7 > delta_min && delta7 < delta_max) delta_max = delta7;

#if 0
		}
#endif
	} else {
		if(delta7 > 0 && delta7 < delta1){
			delta_min = delta7;
			delta_max = delta1;
		} else if(delta7 >= delta1 && delta2 > delta7){
			delta_min = delta1;
			delta_max = delta7;
		} else if(delta7 >= delta2) {
			delta_min = delta2;
			delta_max = delta7;
		} else {
			delta_min = delta1;
			delta_max = delta2;
		}
	}

	if(print_2D_tables){
		if(delta_min < 0 || delta_max < 0){
			fprintf(stderr,"Strange value of delta_min=%d or delta_max=%d.\n",delta_min,delta_max);
		}
	}

	return (100 * delta_min + delta_max);
}
/**********************************************************************************/


/**********************************************************************************/
int MakeDiffTable(void)
/**********************************************************************************/
{
	int i,m,n,m1,n1,mm,nn;
	int xysize = (num_thr + 1) * 11;
	int C1,C2,C3;

	for(i=0;i<xysize;i++) d[i]=0;

	for(m=0;m<=10;m++){
		for(n=0;n<num_thr;n++){
			C2 = C3 = 0;
			for(m1=-1;m1<=1;m1++){
				for(n1=-1;n1<=1;n1++){
					if(m1==0 && n1==0){
						C1 = xy[n*11+m];
						continue;
					}
					if((mm=m+m1) < 0 || mm > 10) continue;
					if((nn=n+n1) < 0 || nn >= num_thr) continue;
					if(m1 == 0 || n1 == 0) C2 += xy[nn*11+mm];
					else C3 += xy[nn*11+mm];
				}
			}
			d[n*11+m] = 4 * C1 - 2 * C2 + C3;
		}
	}
			
	if(print_2D_tables){
		printf("\n");
		printf("Diff table\n");	
		printf("thr     ");
		for(m=0;m<11;m++) printf("%3.1f ",0.1*m);
		printf("\n");
		for(n=0;n<=num_thr;n++){
			printf("%6.1e",thr_list[n]);
			for(m=0;m<11;m++){
				printf("%3d ",d[n*11+m]);
			}
			printf("\n");
		}
		printf("\n");
	}

	return FALSE;
}
/**********************************************************************************/

#define MAX_STACK 50
/**********************************************************************************/
int search_local_maxima(Boolean print_this_table)

/* 351 */
/* This subroutine searches local maxima in the xy table.*/
/* Return value is n*11+m. */
/* 352f10 information content is used to infer optimal border. 
 * Old source was removed.*/
/**********************************************************************************/
{
	int m,n,mm,nn,m1,n1,m2,n2,m_limit;
	int i,i_max;
	static int xy_stack[MAX_STACK][3];
	int flag;
	int best,best_org;
	int largest;
	int delta,delta_min,delta_max,delta_limit;
	int delta7 = -1;
	int minimum_y;

	for(i=0;i<MAX_STACK;i++){
		xy_stack[i][0] = _min_overlap; 
		xy_stack[i][1] = num_thr - 1;
		xy_stack[i][2] = 0;
	}

	largest = 1;
	for(m=0;m<=10;m++){
		for(n=0;n<=num_thr;n++){
			if(largest < xy[n*11+m]){
				largest = xy[n*11+m];
			}
		}
	}
	if(largest > 2) largest = 2;

	if(use_org){
		if((best_org = OptimizeOrg()) <= 0){
			if(print_2D_tables){
				printf("Org information is not used, because ");
				if(best_org == -1) printf("the number of organisms is small.\n");
				else printf("the return value of OptimizeOrg was %d\n",best_org);
				fflush(stdout);
			}
			delta7 = -1;
		} else {
			m2 = best_org % 11;
			n2 = best_org / 11;
			if(m2 < 0 || n2 < 0 || m2 > 10 || n2 > num_thr){
				delta7 = 0;
			} else {
				delta7 = n2 + 10 - m2;
			}

			if(print_2D_tables || print_this_table){
				printf("Optimized point: m2 = %d, n2 = %d, overlap = %3.1f, thr = %6.1e.\n",m2,n2,0.1*m2,thr_list[n2]);
			}
		}
		/* end use_org */	
	}


	/* 352f10, 355r */
	minimum_y = makeInfContTable(1);	/* normally, we use mode 1 */
	delta = selectDeltaFromInf(delta7,minimum_y);

	delta_min = delta / 100;
	delta_max = delta % 100;

	/* 353h */
	if(delta_max - delta_min > _delta_diff) delta_min += _delta_plus;

/* for debug */
	if(print_2D_tables){
		printf("delta_min = %d, delta_max = %d.\n",delta_min,delta_max);
	}

	/* Searching for local maxima */

	i=1;
	/* 352a5 m starts from 4, n ends before num_thr. */
	/* 355x6 Now, searching is done in all area, but only peaks
	 * within the limited area are registered. */
/*	for(m=_min_overlap;m<=10;m++){
*/
	for(m=1;m<=10;m++){
		for(n=0;n<num_thr;n++){
			if(xy[n*11+m] == 0) continue;
			flag = 1;
			for(m1=-1;m1<=1;m1++){
				for(n1=-1;n1<=1;n1++){
					if(m1==0 && n1==0) continue;
/*					if((mm=m+m1) < _min_overlap || mm > 10) continue;
*/
					if((mm=m+m1) < 0 || mm > 10) continue;
					if((nn=n+n1) < 0 || nn >= num_thr) continue;
					if(xy[nn*11+mm] > xy[n*11+m]){
						flag = 0;
						break;
					}
				}
				if(flag==0){
					break;
				}
			}
			if(flag==1){
/*				if(xy[n*11+m] < largest) continue;
 */
				if(m < _min_overlap) continue;
				xy_stack[i][0] = m;
				xy_stack[i][1] = n;
				xy_stack[i][2] = xy[n*11+m];
				i++;
				if(i>=MAX_STACK) break;
			}
		}
		if(i>=MAX_STACK) break;
	}

	if(i>=MAX_STACK) i = MAX_STACK;

	i_max = i;

	if(i_max==1){
		m = xy_stack[0][0];	/* if no local maximum is found, m = 0 */
		n = xy_stack[0][1];	/* n = num_thr */
	} else {
		best = 0;
/*		delta_limit = (-1) - delta_min;
*/
		delta_limit = -1;
		for(i=0;i<i_max;i++){
			if(xy_stack[i][2] > 0 && (delta = 10 + xy_stack[i][1] - xy_stack[i][0]) <= delta_min && delta >= 0){
			   	if(delta_limit < delta){
					delta_limit = delta;
					m_limit = xy_stack[i][0];
					best = i;
				} else if(delta_limit == delta){
					if(m_limit < xy_stack[i][0]){
						m_limit = xy_stack[i][0];
						best = i;
					}
				}
			}
		}
		if(print_2D_tables){
			if(best == 0) fprintf(stderr,"best local maximum was not detected.\n");
		}
		m = xy_stack[best][0];
		n = xy_stack[best][1];
		if(print_2D_tables || print_this_table){
			printf("best = %d, m = %d, n = %d.\n",best,m,n);
		}
	}

	return 1000*delta_max+n*11+m;
}
/**********************************************************************************/


/**********************************************************************************/
int print_2D_matrix(Node **a,unsigned nodes,unsigned i)

/* 351 */
/* This subroutine prints a 2D matrix that represents distribution of score and overlap
 * within the matchlist7 for node i. */
/* 355: Table is always printed for small proteins. */
/**********************************************************************************/
{
	int j,k,m,n;
	unsigned n7;
	double score;
	double overlap;
	int best;
	int xysize = 11*(num_thr+1);
	unsigned target;
	int delta_max;
	double highest_score;
	Boolean print_this_table = FALSE;
	int delta_plus_bak;

	for(m=0;m<11;m++){
		for(n=0;n<=num_thr;n++){
			xy[n*11+m] = 0;
		}
	}

	if(use_org){
		for(m=0;m<=xysize;m++){
			for(n=0;n<=num_org_all;n++){
				sp[n*xysize+m] = 0;
			}
		}
	}

	print_this_table = FALSE;

	n7 = a[i]->n7;
	for(j=0;j<n7;j++){
		score = a[i]->matchlist7[j].score;
		overlap = a[i]->matchlist7[j].overlap;
		if(overlap >= 1.0) m = 10;
		else if(overlap >= 0.9) m = 9;
		else if(overlap >= 0.8) m = 8;
		else if(overlap >= 0.7) m = 7;
		else if(overlap >= 0.6) m = 6;
		else if(overlap >= 0.5) m = 5;
		else if(overlap >= 0.4) m = 4;
		else if(overlap >= 0.3) m = 3;
		else if(overlap >= 0.2) m = 2;
		else if(overlap >= 0.1) m = 1;
		else if(overlap >= 0.0) m = 0;
		else {
			printf("Strange value of overlap in print_2D_matrix with i=%u, j=%d.\n",i,j);
			printf("overlap is made 0.0.\n");
			m = 0;
		}

		for(k=0;k<=num_thr;k++){
			if(score <= thr_list[k]){
				  n	= k;
				  break;
			}
		}
		if(k>=num_thr) n = num_thr;

		xy[n*11+m] += 1;
		if(use_org){
			target = FindNode(a,nodes,a[i]->matchlist7[j].ID);
			if(target == 0 || target > nodes) continue;
			sp[(xysize * a[target]->species)+(n*11+m)] += 1;
		}
	
		if(a[i]->matchlist7[j].ID == a[i]->n0){
			highest_score = a[i]->matchlist7[j].score;

/*			if(highest_score > _MDR_score2) print_this_table = TRUE;
			else print_this_table = FALSE;
for development in 355			*/
		}
	}

	if(print_2D_tables || print_this_table){

		printf("\n");
		printf("2D table for node %u %s\n",i,a[i]->name);	
		printf("thr     ");
		for(m=0;m<11;m++) printf("%3.1f ",0.1*m);
		printf("\n");

		for(n=0;n<=num_thr;n++){
			printf("%6.1e",thr_list[n]);
			for(m=0;m<11;m++){
				printf("%3d ",xy[n*11+m]);
			}
			printf("\n");
		}
		printf("\n");

	}

	/* 355q */
	delta_plus_bak = _delta_plus;
	if(a[i]->len < _MDR_length2) _delta_plus += 1;
	else if(a[i]->len < _MDR_length1) _delta_plus += 2;

	only1_flag = 0;		/* 352f10 */
	best = search_local_maxima(print_this_table);
	if(best<=0) {
		delta_max = num_thr + 10 - _min_overlap;
		m = 0;
		n = 0;
	} else {
		delta_max = best / 1000;
		m = (best % 1000) % 11;
		n = (best % 1000) / 11;
	}
	/* 355y */
	a[i]->delta_max = delta_max;

	/* 355r */
	if(a[i]->len < _MDR_length2) delta_max += _delta_plus;

	_delta_plus = delta_plus_bak;

	/* for security */

	if(n < 0) n = 0;
   	else if(n > num_thr) n = num_thr;
	
	if(print_2D_tables || print_this_table){
		printf("Best local maximum: m = %d, n = %d, overlap = %3.1f, thr = %6.1e, delta_max = %d.\n",\
				m,n,0.1*m,thr_list[n],delta_max);
		fflush(stdout);


/* Not used now. 352f7
		MakeDiffTable();
		printf("\n");
		fflush(stdout);
*/
	}

	keep_data = 0;
	return select_initial_clique(m,n,delta_max);
	
}
/**********************************************************************************/


/**********************************************************************************/
int best_thr(SQlist *s, unsigned n)

/* 350 */
/* In this function, 
 1. distribution of E-values within the sqlist3 is evaluated according to the 
    thr levels defined by var_list,
 2. the thr value that gives the minimum count in the distribution is selected,
 3. and returns the thr level as an integer.*/
/**********************************************************************************/
{
	/* globals used in this function: num_thr, thr_list[],number_of_genomes */

	static unsigned counts[50];
	int i,j;
	double e;
	int min,max;
	unsigned cumulative_count = 0;

	/* initialization */

	for(i=0;i<50;i++) counts[i]=0;

	/* part 1 */

	for(j=0;j<n;j++){
		e = s[j].score;
		for(i=0;i<num_thr;i++){
			if(e < thr_list[i]) break;
		}
		counts[i]++;
	}

/* debug 
printf("E-value list\n");
for(j=0;j<n;j++) printf("%8.2e ",s[j].score);
printf("\n");
for(i=0;i<num_thr;i++) printf("%u ",counts[i]);
printf("\n");
*/

	/* part 2 */
	
	min = num_thr;
	max = 0;

	/* if thr levels are not defined, */

	if(num_thr <= 2) return min;

	/* Detection of the first peak */

	for(i=1;i<num_thr;i++){
		if(counts[i-1] > 1 && counts[i-1] > counts[i]) break;
	}
	if(i== num_thr){		/* monotonous increase */
		cumulative_count = 0;
		for(i=0;i<num_thr;i++){
			cumulative_count += counts[i];
			if(cumulative_count > number_of_genomes) break;
		}		
		min = i;
	} else {
		max = i;
		for(i=max;i<num_thr;i++){
			if(counts[i-1] < counts[i] || counts[i] == 0) break;
		}
		min = i;
	}

/* debug 
printf("min=%d, max=%d\n",min,max);
*/

	return min;

}
/**********************************************************************************/


/**********************************************************************************/
int tTest_idlist4(Node **a,unsigned nodes,unsigned i,unsigned *vector,unsigned dim)
/* returns TRUE if the new list4 should not be adopted. */
/* This is based on comparison of variation of old and new idlist4. */
/**********************************************************************************/
{
	Av_Sw *avsw1;
	Av_Sw *avsw2;
	double Vw,t;
	int freedom;
	unsigned j,k,jj;
	unsigned n4b;
	unsigned real_n4a=0;
	unsigned real_n4b=0;

	n4b=a[i]->n4b;


printf("List b for i=%u, %s\n",i,a[i]->name);

	jj=0;
	for(j=0;j<n4b;j++){
		k = FindNode(a,nodes,a[i]->idlist4b[j]);
		if(k==0 || a[k]->len == 0) continue;
		if(dim < jj) {
			fprintf(stderr,"dim = %u is too small in tTest 1.\n",dim);
			return FALSE;
		}	
		vector[jj] = a[k]->len;


printf("%u ",vector[jj]);


		jj++;
	}
	real_n4b = jj;
	
printf("\n");
fflush(stdout);

	avsw1 = GetAvSwU(vector,real_n4b);

/* abort judging if deviation in 4b is large */
	if(real_n4b > 3){
		if(avsw1->Sw / (real_n4b - 1) > 0.5 * avsw1->average) return TRUE;
	}

printf("List a for i=%u\n",i);


	jj=0;	
	for(j=0;j<a[i]->n4;j++){
		for(k=0;k<n4b;k++){
			if(a[i]->idlist4[j] == a[i]->idlist4b[k]) break;
		}
		if(k==n4b) {
			k = FindNode(a,nodes,a[i]->idlist4[j]);
			if(k==0 || a[k]->len == 0) continue;
			if(dim < jj) {
				fprintf(stderr,"dim = %u is too small in tTest 2.\n",dim);
				return FALSE;
			}	
			vector[jj] = a[k]->len;


printf("%u ",vector[jj]);


			jj++;
		}
	}
	real_n4a = jj;
	
	
printf("\n");
fflush(stdout);

	avsw2 = GetAvSwU(vector,real_n4a);

	freedom = real_n4a + real_n4b - 2;
	if(freedom < 1) return FALSE;	/* unable to judge */
	t = fabs(avsw1->average - avsw2->average);

	
printf("n4a=%u, n4b=%u, freedom=%d, diff_average=%f, ",real_n4a,real_n4b,freedom,t);


	Vw = (avsw1->Sw + avsw2->Sw) / (double)freedom;


printf("Vw=%e, ",Vw);


	Vw = (1.0/(double)real_n4a + 1.0/(double)real_n4b) * Vw;
	if(Vw < (avsw1->average + avsw2->average) / 2.0) Vw = (avsw1->average + avsw2->average) / 4.0;
	t = t / sqrt(Vw);

printf("t=%f, tTest=%f.\n",t,T_dist5(freedom));

	if(T_dist5(freedom) < t) return TRUE;
	else return FALSE;
}
/**********************************************************************************/

/*******************************************/
int isMultiDomain(unsigned long domainID)
/*******************************************/
{
	unsigned counter=0;
	unsigned n;
	unsigned long domainS=0x01;
	int md=0;

	for(n=0;n<24;n++){
		if(domainID & domainS) counter += 1;
		domainID >>= 1;
	}

	if(counter > 1){
		md = TRUE;
	}

	return md;
}
/*******************************************/

/****************************************************************/
unsigned overlap(unsigned a1,unsigned a2,unsigned b1,unsigned b2)
/****************************************************************/
{
	unsigned x=0;
	unsigned tmp;

	if(a1>a2) {
		tmp=a1;
		a1=a2;
		a2=tmp;
	}
	if(b1>b2) {
		tmp=b1;
		b1=b2;
		b2=tmp;
	}

	if(a1 <= b1 && b1 <= a2){
		if(b2 > a2) x=a2-b1;
		else x=b2-b1;
	} else if(a1 <= b2 && b2 <= a2){
		x=b2-a1;
	} else if(a2 <= b2 && b1 <= a1){
		x=a2-a1;
	}

	return x;
}
/****************************************************************/

/****************************************************************/
DMlr *OverlapDomains(unsigned a1,unsigned a2,unsigned b1,unsigned b2)
/****************************************************************/
{
	DMlr *dmlr_1;

	if((dmlr_1=(DMlr*)calloc(1,sizeof(DMlr)))==NULL){
		fprintf(stderr,"Error in allocating memory in OverlapDomains1.\n");
		exit(1);
	}

	dmlr_1->Al=0;
	dmlr_1->Ar=0;
	dmlr_1->Bl=0;
	dmlr_1->Br=0;
	dmlr_1->overlap=0;

	if(a1==a2 || b1==b2) return dmlr_1;

	if(a1>a2) swapUnsigned(&a1,&a2);
	if(b1>b2) swapUnsigned(&b1,&b2);

	if(b1 <= a1 && a1 <= b2 && b2 <= a2){
		dmlr_1->Al = 0;
		dmlr_1->Ar = a2 - b2;
		dmlr_1->Bl = a1 - b1;
		dmlr_1->Br = 0;
		dmlr_1->overlap = b2 - a1;
	} else if(a1 <= b1 && b1 <= a2 && a2 <= b2){
		dmlr_1->Al = b1 - a1;
		dmlr_1->Ar = 0;
		dmlr_1->Bl = 0;
		dmlr_1->Br = b2 - a2;
		dmlr_1->overlap = a2 - b1;
	} else if(b1 <= a1 && a1 <= a2 && a2 <= b2){
		dmlr_1->Al = 0;
		dmlr_1->Ar = 0;
		dmlr_1->Bl = a1 - b1;
		dmlr_1->Br = b2 - a2;
		dmlr_1->overlap = a2 - a1;
	} else if(a1 <= b1 && b1 <= b2 && b2 <= a2){
		dmlr_1->Al = b1 - a1;
		dmlr_1->Ar = a2 - b2;
		dmlr_1->Bl = 0;
		dmlr_1->Br = 0;
		dmlr_1->overlap = b2 - b1;
	}
	
	return dmlr_1;
}
/****************************************************************/


/****************************************************************/
double OverlapScore(unsigned a1,unsigned a2,unsigned b1,unsigned b2)
/****************************************************************/
{
	unsigned x;
	double y=0.0;
	unsigned tmp;
	unsigned index;

	if(a1>a2) {
		tmp=a1;
		a1=a2;
		a2=tmp;
	}
	if(b1>b2) {
		tmp=b1;
		b1=b2;
		b2=tmp;
	}

	if(a1 <= b1 && b1 <= a2){
		if(b2 > a2) x=a2-b1;
		else x=b2-b1;
	} else if(a1 <= b2 && b2 <= a2){
		x=b2-a1;
	} else if(b1 <= a1 && a2 <= b2){
		x=a2-a1;
	} else x=0;

	if(a2  + b2 - a1 - b1 == 0) y = 0.0;
	else y = 2.0 * (double)x /(double)(a2  + b2 - a1 - b1);

	index = (unsigned)(y / 0.050);
	if(index >= 0 && 20 >= index){
		dist_score[index] += 1;
		num_score += 1;
	}
	
	return y;
}
/****************************************************************/


/***************************************************************/
unsigned GetRegion(SQlist *s,unsigned j2,unsigned d)
/***************************************************************/
{
	unsigned L1,L2,t2;
	unsigned tmp;
	unsigned lengthS;
	unsigned a1,a2,b1,b2;

	L2=0;
	for(L1=0;L1<d;L1++){
		if(s[L1].ID == s[j2].ID && s[L1].qID == s[j2].qID){
			R[L2].start = s[L1].Sstart;
			R[L2].end = s[L1].Send;
			if(R[L2].end < R[L2].start){
				tmp = R[L2].end;
				R[L2].end = R[L2].start;
				R[L2].start = tmp;
			}
			L2++;
		}
	}
	t2 = L2;
	if(t2 > 1){
		for(L1=1;L1<t2;L1++){
			if(R[L1].start == 0) continue;
			for(L2=0;L2<L1;L2++){
				if((b1=R[L2].start)==0) continue;
				b2=R[L2].end;
				a1=R[L1].start;
				a2=R[L1].end;
				
				if(a1 < b1 && b1 < a2){
					if(a2 < b2) R[L1].end = b2;
				} else if(b1 < a1 && a1 < b2){
					R[L1].start = b1;
					if(a2 < b2) R[L1].end = b2;
/* fix Sept 6 */
				} else continue;
				R[L2].start = 0;
				R[L2].end = 0;
			}
		}
		lengthS = 0;
		for(L1=0;L1<t2;L1++) lengthS += R[L1].end - R[L1].start;

	} else {
		lengthS = R[0].end - R[0].start;
	}

	return lengthS;
}
/*************************************************************/


/*****************************************************************************/
unsigned GetRegion2(SQlist *s,unsigned t1,SQlist *t,unsigned t2)
/* this is to get overlaped region of two sequences on a common sequence */
/*****************************************************************************/
{
	unsigned L1,L2,L3,t3;
	unsigned tmp;
	unsigned lengthS;
	unsigned a1,a2,b1,b2;

	if(t1==0 || t2==0){
		fprintf(stderr,"Zero items in GetRegion2.\n");
		return 0;
	}
	if(s==NULL || t==NULL){
		fprintf(stderr,"Null input in GetRegion2.\n");
		return 0;
	}
	for(L1=0;L1<t1;L1++){
		R1[L1].start = s[L1].Qstart;
		R1[L1].end = s[L1].Qend;
		if(R1[L1].end < R1[L1].start){
			tmp = R1[L1].end;
			R1[L1].end = R1[L1].start;
			R1[L1].start = tmp;
		}
	}
	for(L1=0;L1<t2;L1++){
		R2[L1].start = s[L1].Qstart;
		R2[L1].end = s[L1].Qend;
		if(R2[L1].end < R2[L1].start){
			tmp = R2[L1].end;
			R2[L1].end = R2[L1].start;
			R2[L1].start = tmp;
		}
	}

	if(t1==1 && t2==1){
		a1=R1[0].start;
		a2=R1[0].end;
		b1=R2[0].start;
		b2=R2[0].end;
		return overlap(a1,a2,b1,b2);
	}
	if(t1>1){
		for(L1=1;L1<t1;L1++){
			if(R1[L1].start == 0) continue;
			for(L2=0;L2<L1;L2++){
				if((b1=R1[L2].start)==0) continue;
				b2=R1[L2].end;
				a1=R1[L1].start;
				a2=R1[L1].end;
				
				if(a1 < b1 && b1 < a2){
					if(a2 < b2) R1[L1].end = b2;
				} else if(b1 < a1 && a1 < b2){
					R1[L1].start = b1;
					if(a2 < b2) R1[L1].end = b2;
				} else continue;
				R1[L2].start = 0;
				R1[L2].end = 0;
			}
		}
	}

	if(t2>1){
		for(L1=1;L1<t2;L1++){
			if(R2[L1].start == 0) continue;
			for(L2=0;L2<L1;L2++){
				if((b1=R2[L2].start)==0) continue;
				b2=R2[L2].end;
				a1=R2[L1].start;
				a2=R2[L1].end;
				
				if(a1 < b1 && b1 < a2){
					if(a2 < b2) R2[L1].end = b2;
				} else if(b1 < a1 && a1 < b2){
					R2[L1].start = b1;
					if(a2 < b2) R2[L1].end = b2;
				} else continue;
				R2[L2].start = 0;
				R2[L2].end = 0;
			}
		}
	}

/* only this part is different. overlaped region is selected. */

	for(L3=0;L3<t1+t2;L3++){
		R[L3].start = 0;
		R[L3].end = 0;
	}

	L3 = 0;
	for(L1=0;L1<t1;L1++){
		if(R1[L1].start == 0) continue;
		for(L2=0;L2<t2;L2++){
			if((b1=R2[L2].start)==0) continue;
			b2=R2[L2].end;
			a1=R1[L1].start;
			a2=R1[L1].end;
		
			if(a1 <= b1 && b1 <= a2){
				R[L3].start = b1;
				if(b2 <= a2) R[L3].end = b2;
				else R[L3].end = a2;
				L3++;
			} else if(b1 <= a1 && a1 <= b2){
				R[L3].start = a1;
				if(b2 <= a2) R[L3].end = b2;
				else R[L3].end = a2;
				L3++;
			}
		}
	}
	t3=L3;

/* last step of compaction */

	if(t3==0) return 0;
	else if(t3==1) return (R[0].end - R[0].start);

	for(L1=1;L1<t3;L1++){
		if(R[L1].start == 0) continue;
		for(L2=0;L2<L1;L2++){
			if((b1=R[L2].start)==0) continue;
			b2=R[L2].end;
			a1=R[L1].start;
			a2=R[L1].end;
			
			if(a1 < b1 && b1 < a2){
				if(a2 < b2) R[L1].end = b2;
			} else if(b1 < a1 && a1 < b2){
				R[L1].start = b1;
				if(a2 < b2) R[L1].end = b2;
			} else continue;
			R[L2].start = 0;
			R[L2].end = 0;
		}
	}

	lengthS = 0;
	for(L1=0;L1<t3;L1++) lengthS += R[L1].end - R[L1].start;

	return lengthS;
}
/*************************************************************/

/***************************************************************/
int MatchIDlist(Node **a,unsigned nodes,unsigned i, unsigned j)
/***************************************************************/
{
	unsigned x4,y4,k,m;
	
	if(i > nodes || j > nodes){
		fprintf(stderr,"index i=%u or j=%u is larger than nodes.\n",i,j);
		return FALSE;
	}
	x4 = a[i]->n4;
	y4 = a[j]->n4;

	if(x4 != y4) return TRUE;
	for(k=0;k<x4;k++){
		for(m=0;m<y4;m++){
			if(a[i]->idlist4[k] == a[j]->idlist4[m]) break;
		}
		if(m==y4) return TRUE;
	}

	return FALSE;
}
/***************************************************************/


/***************************************************************/
int GetScore(Node **a,unsigned nodes,unsigned i, unsigned k)
/* i and k should be the current node numbers. */
/***************************************************************/
{	
	unsigned j;
	int x=0;
	double real_thr=thr;
	unsigned pID,ix;

	if(i==0 || k==0) return 0;
	if(i > nodes || k > nodes){
		fprintf(stderr,"index i=%u or k=%u is larger than nodes.\n",i,k);
		return 0;
	}
	for(j=0;j<a[i]->n1b;j++){
						/* 350 */
		if(clique_mode){
			if((pID = a[i]->pID) == 0) real_thr=a[i]->final_thr;
			else {
				ix = FindNode(a,nodes,pID);
				if(ix == 0) real_thr=a[i]->final_thr;
				else real_thr = a[ix]->final_thr;
			}
		} else if(a[i]->final_thr < thr) real_thr=a[i]->final_thr;

		if(a[i]->sqlist1[j].ID == a[k]->n0){
			if(a[i]->sqlist1[j].score >= real_thr) x=0;
			else x = 1000;
			break;
		}
	}
	
	return x;
}
/***************************************************************/


/********************************************************************************/
double OverlapScore2(SQlist *s,unsigned t2,unsigned len1,unsigned len2)
/********************************************************************************/
{
	unsigned L1,L2;
	double value;
	unsigned tmp;
	unsigned lengthS;
	unsigned lengthQ;
	unsigned a1,a2,b1,b2;

	if(s==NULL){
		fprintf(stderr,"Strange input for s in OverlapScore2 1.\n");
		return 0.0;
	} else if(R==NULL){
		fprintf(stderr,"Strange input for R in OverlapScore2 2.\n");
		return 0.0;
	}

	if(t2==0 || len1==0 || len2==0) {
		return 0.0;
	}

	if(t2 > 1){
		for(L1=0;L1<t2;L1++){
			R[L1].start = (s+L1)->Sstart;
			R[L1].end = (s+L1)->Send;
			if(R[L1].end < R[L1].start){
				tmp = R[L1].end;
				R[L1].end = R[L1].start;
				R[L1].start = tmp;
			}
		}
		for(L1=1;L1<t2;L1++){
			if(R[L1].start == 0) continue;
			for(L2=0;L2<L1;L2++){
				if((b1=R[L2].start)==0) continue;
				b2=R[L2].end;
				a1=R[L1].start;
				a2=R[L1].end;
				
				if(a1 < b1 && b1 < a2){
					if(a2 < b2) R[L1].end = b2;
				} else if(b1 < a1 && a1 < b2){
					R[L1].start = b1;
					if(a2 < b2) R[L1].end = b2;
				} else continue;
				R[L2].start = 0;
				R[L2].end = 0;
			}
		}
		lengthS = 0;
		for(L1=0;L1<t2;L1++) lengthS += (R[L1].end - R[L1].start + 1);

		for(L1=0;L1<t2;L1++){
			R[L1].start = (s+L1)->Qstart;
			R[L1].end = (s+L1)->Qend;
			if(R[L1].end < R[L1].start){
				tmp = R[L1].end;
				R[L1].end = R[L1].start;
				R[L1].start = tmp;
			}
		}
		for(L1=1;L1<t2;L1++){
			if(R[L1].start == 0) continue;
			for(L2=0;L2<L1;L2++){
				if((b1=R[L2].start)==0) continue;
				b2=R[L2].end;
				a1=R[L1].start;
				a2=R[L1].end;
				
				if(a1 < b1 && b1 < a2){
					if(a2 < b2) R[L1].end = b2;
				} else if(b1 < a1 && a1 < b2){
					R[L1].start = b1;
					if(a2 < b2) R[L1].end = b2;
				} else continue;
				R[L2].start = 0;
				R[L2].end = 0;
			}
		}
		lengthQ = 0;
		for(L1=0;L1<t2;L1++) lengthQ += (R[L1].end - R[L1].start + 1);
	} else {
		lengthS = s[0].Send - s[0].Sstart + 1;
		lengthQ = s[0].Qend - s[0].Qstart + 1;
	}
	value = (double)(lengthS + lengthQ)/(double)(len1 + len2);

	return value;
}
/*********************************************************/


/***************************************************************/
Region *ReallocRegion(Region *R, unsigned i1, unsigned i2)
/***************************************************************/
{
	unsigned j;
	unsigned size = i2+1;

	if(i1 > i2) return R;
	if((R=(Region*)realloc(R,size*sizeof(Region)))==NULL){
		fprintf(stderr,"Error in ReallocRegion1.\n");
		exit(1);
	}
	for(j=0;j<size;j++){
		R[j].start=0;
		R[j].end=0;
	}
	
	return R;
}
/***************************************************************/


/************************************************************/
void swapSQlist(SQlist *a,SQlist *b)
/************************************************************/
{
	static SQlist tmp;

/*
	if((tmp=(SQlist*)calloc(1,sizeof(SQlist)))==NULL){
		fprintf(stderr, "Memory allocation error in swapSQlist\n.");
		exit(1);
	}
*/
	copySQlist(&tmp,a);
	copySQlist(a,b);
	copySQlist(b,&tmp);

	return;
}
/************************************************************/

/************************************************************/
void swapUnsigned(unsigned *a,unsigned *b)
/************************************************************/
{
	unsigned tmp;

	tmp = *a;
	*a = *b;
	*b = tmp;

	return;
}
/************************************************************/

/************************************************************/
void swapInt(int *a,int *b)
/************************************************************/
{
	int tmp;

	tmp = *a;
	*a = *b;
	*b = tmp;

	return;
}
/************************************************************/


/***************************************************************/
void ClearRegion(Region *RR,unsigned sizeRegion)
/***************************************************************/
{
	unsigned j;

	for(j=0;j<sizeRegion;j++){
		RR[j].start=0;
		RR[j].end=0;
	}
	return;
}
/***************************************************************/


/***************************************************************/
Region *InitRegion(unsigned i)
/***************************************************************/
{
	Region *R;
	unsigned j;
	unsigned size = i+1;

	if((R = (Region*)calloc(size,sizeof(Region)))==NULL){
		fprintf(stderr,"Error in InitRegion2.\n");
		exit(1);
	}
	for(j=0;j<size;j++){
		R[j].start=0;
		R[j].end=0;
	}
	
	return R;
}
/***************************************************************/


/***************************************************************/
char *c10_2(unsigned long data, unsigned bits)
/***************************************************************/
{
	#define BITS 128	/* upper limit of bits */
	static char b[BITS+BITS/4];	/* bit array */
	int i, j;

	if(bits > BITS) bits=BITS;
	for(j=0,i=bits-1;i >= 0;i--,j++){
		if((data >> i) & 0x01){
			b[j]='1';
		} else {
			b[j]='0';
		}
		if(i%4==0 && i!=bits && i!=0) b[++j]=' ';
	}
	b[j]=0;
	return b;
}
/***************************************************************/


/***************************************************************/
char *c10_3(unsigned long data, unsigned long data2, unsigned bits, char x, char y)
/***************************************************************/
{
	#define BITS 128	/* upper limit of bits */
	static char b[BITS+BITS/4];	/* bit array */
	int i, j;

	if(bits > BITS) bits=BITS;
	for(j=0,i=bits-1;i >= 0;i--,j++){
		if((data >> i) & 0x01){
			b[j]= x;
		} else if((data2 >> i) & 0x01){
			b[j]= y;
		} else {
			b[j]='.';
		}
		if(i%4==0 && i!=bits && i!=0) b[++j]=' ';
	}
	b[j]=0;
	return b;
}
/***************************************************************/
/*    end of file anal.c    */
